<?php

namespace App\Exports;

use App\Models\Communication;
use App\Models\Event;
use App\Models\Lead;
use DateTime;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;

class AllLeadsExport implements FromCollection, WithHeadings
{
    protected $leads;

    public function __construct($leads, )
    {
        $this->leads = $leads;
    }

    public function collection()
    {
        return $this->leads->map(function ($lead) {

            $phoneNumbers = [];
            foreach ($lead->phone_numbers as $phoneNumber) {
                $phoneNumbers[] = $phoneNumber->country->phonecode . ' ' . $phoneNumber->phone;
            }

            // Retrieve only the last two comments and map them to the description
            $comments = $lead->communications->sortByDesc('created_at')->take(3)
                ->pluck('description')->toArray();

            // Make sure to pad the array so it always has two elements
            $paddedComments = array_pad($comments, 3, '--');


            return [
                $lead->name,
                ...array_pad($phoneNumbers, 2, '--'),
                $lead->user?->name ?? '', // Assigned To
                ...$paddedComments, // Last Comments
                $lead->communications->last()?->created_at->format('Y-m-d H:i') ?? 'not yet', // Last Action Date
                $this->getLastEventDate($lead->id) ?? '--', // next action date
                $lead->LeadStatus?->name ?? '', // Status
                $lead->note ?? '', // Note
                $lead->target ?? '', // Lead Target
                $lead->source?->sourceType?->name ?? '', // Source Type
                $lead->campaign?->name ?? '', // Campaign
                $lead->project?->name ?? '', // Project
                $lead->tag ?? '', // Tag
                $lead->created_by_user?->name ?? '', // Created By
                $lead->created_at->format('Y-m-d'), // Created Date
                $lead->created_at->format('H:i'), // Created Time
            ];
        });
    }

    public function headings(): array
    {
        return [
            'Name',
            'Phone Number 1',
            'Phone Number 2',
            'Assigned To',
            'Last Comment 1',
            'Last Comment 2',
            'Last Comment 3',
            'Last Action Date',
            'Next Action Date',
            'Status',
            'Note',
            'Lead Target',
            'Source Type',
            'Campaign',
            'Project',
            'Tag',
            'Created By',
            'Created Date',
            'Created Time',
        ];
    }

    protected function getLastEventDate($leadId)
    {
        // Initialize the variable to hold the last event date
        $lastEventStart = null;

        $communications = Communication::where('lead_id', $leadId)->first();

        if ($communications) {
            $event = Event::where('communication_id', $communications->id)->first();
            if ($event) {
                $lastEventStart = $event->start;
                $dateString = substr($lastEventStart, 0, 10);
                $dateTime = new DateTime($dateString);
                $formattedDate = $dateTime->format('Y-m-d');
            } else {
                return ['--'];
            }
        } else {
            return ['--'];
        }


        // Return the formatted date or a default value
        return $formattedDate;
    }
}
