<?php

namespace App\Http\Controllers\Administration;

use App\Exports\AllLeadsExport;
use Exception;
use Carbon\Carbon;
use App\Models\Lead;
use App\Models\User;
use App\Models\Role;
use App\Models\Event;
use App\Models\Broker;
use App\Models\Source;
use App\Models\Status;
use App\Models\Country;
use App\Models\Inquiry;
use App\Models\Project;
use App\Models\Setting;
use App\Models\Campaign;
use App\Models\LeadUser;
use App\Models\ActivityLog;
use App\Models\PhoneNumber;
use App\Imports\LeadsImport;
use Illuminate\Http\Request;
use App\Models\Communication;
use App\Http\Requests\LeadRequest;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Requests\UploadLeadRequest;

class LeadsController extends Controller
{
    public function __construct()
    {
        // $this->middleware('permission:leads-create', ['only' => ['create', 'store']]);
        // $this->middleware('permission:leads-read', ['only' => ['show']]);
        // $this->middleware('permission:leads-update', ['only' => ['edit', 'update']]);
        // $this->middleware('permission:leads-delete', ['only' => ['destroy']]);
        // $this->middleware('permission:leads-export', ['only' => ['export']]);
        // $this->middleware('permission:leads-import', ['only' => ['import']]);
        // $this->middleware('permission:leads-ChangeStatus',['only' => ['ChangeStatus']]);
        // $this->middleware('permission:leads-reAssignUser',['only' => ['reAssignUser']]);
    }

    public function index(Request $request, $status = null)
    {
        $leads = Lead::query();
        switch ($status) {
            case 'fresh':
                $leads = $leads->where('status_id', 1);
                break;
            case 'interested':
                $leads = $leads->where('status_id', 20);
                break;
            case 'fakeNumber':
                $leads = $leads->where('status_id', 21);
                break;
            case 'meeting':
                $leads = $leads->where('status_id', 27);
                break;
            case 'pending':
                $leads = $leads->where('status_id', 28);
                break;
            case 'followUp':
                $leads = $leads->where('status_id', 29);
                break;
            case 'lowBudget':
                $leads = $leads->where('status_id', 30);
                break;
            case 'noRegisteration':
                $leads = $leads->where('status_id', 31);
                break;
            case 'differentLocation':
                $leads = $leads->where('status_id', 32);
                break;
            case 'closed':
                $leads = $leads->where('status_id', 36);
                break;
            case 'callFailed':
                $leads = $leads->where('status_id', 37);
                break;
            case 'noAnswer':
                $leads = $leads->where('status_id', 38);
                break;
        }
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Marketing Team Leader' || auth()->user()->role->name == 'Sales Admin') {
            $leads->where(['is_delay' => 0, 'type' => 'lead']);
        } elseif (auth()->user()->assigned_to == null) {
            // $childUsers = auth()->user()->childs()->pluck('id')->toArray();
            $childUsers = auth()->user()->allChildren()->pluck('id')->toArray();
            if (count($childUsers) > 0) {
                $leads = Lead::where(function ($query) {
                    $query->where('created_by', auth()->user()->id)
                        ->orWhereIn('user_id', auth()->user()->allChildren()->pluck('id')->toArray());
                });
            } else {
                $leads = Lead::where('created_by', auth()->user()->id);
            }
        } else {
            $users = User::all();
            $CurrentUser = auth()->user();
            $UserChildren = array();

            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }

            $LeadBulk = array();
            if (count($CurrentUser->leads) != 0) {
                foreach ($CurrentUser->leads as $lead) {
                    array_push($LeadBulk, $lead->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $sales = User::find($UserChildren[$i]['id']);
                if (count($sales->leads) == 0) {
                    continue;
                } else {
                    foreach ($sales->leads as $lead) {
                        array_push($LeadBulk, $lead->id);
                    }
                }
            }
            $LeadBulk = array_unique($LeadBulk);
            // $leads = Lead::query();
            $leads->where('is_delay', 0);
            $leads->whereIn('id', $LeadBulk);
        }

        // Filters variables
        $statuses = Status::get();
        $statusTypes = DB::table('statuses')->pluck('type')->toArray();
        $statusTypes = array_unique($statusTypes);
        $users = $creators = User::where('id', '!=', 1)->get();
        $campaigns = Campaign::all();
        $sources = Source::all();
        $projects = Project::all();
        $events = Event::with('communications')->get();
        $countries = Country::get();


        // Filter query 
        if ($request->user_id) {
            $leads->whereIn('user_id', $request->user_id);
        }

        if ($request->creator_id) {
            $leads->whereIn('created_by', $request->creator_id);
        }

        if ($request->status_id) {
            $leads->whereIn('status_id', $request->status_id);
        }

        if ($request->statusTypes) {
            $types = Status::whereIn('type', $request->statusTypes)->pluck('id')->toArray();
            $leads->whereIn('status_id', $types);
        }

        if ($request->contact_tool) {
            $leads->whereIn('contact_tool', $request->contact_tool);
        }

        if ($request->campaign_id) {
            $leads->whereIn('campaign_id', $request->campaign_id);
        }

        if ($request->project_id) {
            $leads->whereIn('project_id', $request->project_id);
        }

        if ($request->country_id) {
            $lead_id = PhoneNumber::where('country_id', $request->country_id)->where('lead_id', '!=', NULL)->pluck('lead_id')->toArray();
            $leads->whereIn('id', $lead_id);
        }

        if ($request->source_id) {
            $leads->whereIn('source_id', $request->source_id);
        }

        if ($request->reply_option) {
            $leads->whereIn('reply_option', $request->reply_option);
        }

        if ($request->purpose) {
            $leads->whereIn('purpose', $request->purpose);
        }

        if ($request->segment) {
            $leads->whereIn('segment', $request->segment);
        }

        if ($request->tag) {
            $leads->whereIn('tag', $request->tag);
        }

        if ($request->time1 && $request->time2) {
            $leads->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->name) {
            $leads->where('name', 'LIKE', '%' . $request->name . '%');
        }

        if ($request->phone) {
            $leads_id = PhoneNumber::where('phone', 'LIKE', '%' . $request->phone . '%')->where('lead_id', '!=', null)->pluck('lead_id')->toArray();
            $leads->whereIn('id', $leads_id);
        }

        if ($request->lastActio_start_date && $request->lastActio_end_date) {
            $communications = Communication::query();
            $communications->whereDate('created_at', '>=', $request->lastActio_start_date)
                ->whereDate('created_at', '<=', $request->lastActio_end_date);

            $communications = $communications->get();
            $communicationIds = $communications->pluck('lead_id')->toArray();

            $leads->whereIn('id', $communicationIds);
        }

        if ($request->date1 && $request->date2) {
            $leads->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }
        $leads = $leads->with('communications')->paginate(50)->withQueryString();

        // Specific permissions for roles 
        $CurrentUser = auth()->user();
        if ($CurrentUser->type == 'admin' || $CurrentUser->role->name == 'Marketing Team Leader' || $CurrentUser->role->name == 'Sales Admin') {
            $UserChildren = $users;
        } else {
            $UserChildren = array();
            foreach ($users as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }
        }
        // Max Number of phones
        $leadNumbers = 0;
        foreach ($leads as $lead) {
            $maxCount = count($lead->phone_numbers);
            // dd($maxCount);
            if ($maxCount > $leadNumbers) {
                $leadNumbers = $maxCount;
            }
        }

        // Activity Log
        $model = new Lead();
        // $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Index')
            ->log(auth()->user()->name . ' viewed leads index ');

        return view('leads.index', compact('leads', 'users', 'countries', 'campaigns', 'UserChildren', 'projects', 'sources', 'statuses', 'creators', 'leadNumbers', 'statusTypes', 'events'));
    }

    public function create()
    {
        $countries = Country::all();
        $campaigns = Campaign::all();
        $brokers = Broker::get();
        $users = User::get();
        $CurrentUser = auth()->user();
        if ($CurrentUser->type != 'admin') {
            $UserChildren = array();
            foreach ($users as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }
        } else {
            $UserChildren = $users;
        }

        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Create')
            ->log(auth()->user()->name . ' viewed leads creation page ');

        return view('leads.create', compact('UserChildren', 'countries', 'campaigns', 'brokers', 'users'));
    }

    public function store(Request $request)
    {
        // dd($request);
        try {
            // DB::beginTransaction();
            $lead = new Lead();
            $lead->name = $request->name;
            $lead->email = $request->email ?? null;
            $lead->job_title = $request->job_title ?? null;
            $lead->contact_tool = $request->contact_tool ?? null;
            $lead->reply_option = $request->reply_option ?? null;
            $lead->purpose = $request->purpose ?? null;
            $lead->segment = $request->segment ?? null;
            $lead->tag = $request->tag ?? null;
            $lead->note = $request->note;
            $lead->status_id = $request->status_id;
            $lead->type = $request->type;
            $lead->user_id = $request->user_id ?? null;
            if ($request->type == 'lead') {
                $lead->campaign_id = $request->campaign_id;
                $lead->source_id = $request->source_id;
                $lead->came_from = $request->came_from;
                $campaign = Campaign::find($request->campaign_id);
                if (isset($campaign)) {
                    $lead->project_id = $campaign->project?->id ?? NULL;
                }
            } else {
                if ($request->confirmed == 'on') {
                    $lead->confirmation_date = date("Y-m-d H:i:s");
                    $lead->is_confirmed = 1;
                }
                $lead->civil_id = $request->civil_id;
                $lead->broker_id = $request->broker_id;
                $lead->agent_id = $request->agent_id;
            }
            if ($request->inquiry_id) {
                $lead->inquiry_id = $request->inquiry_id;

                $inquiry = Inquiry::find($request->inquiry_id);
                $inquiry->lead_id = $lead->id;
                $inquiry->save();
            }

            $lead->created_by = auth()->user()->id;
            $lead->save();
            try {
                if ($request->mobile_number) {
                    for ($i = 0; $i < count($request->mobile_number); $i++) {
                        $phone = new PhoneNumber();
                        $phone->lead_id = $lead->id;
                        $phone->country_id = $request->country_id[$i];
                        $phone->phone = $request->mobile_number[$i];
                        $phone->save();
                    }
                }
            } catch (Exception $e) {
                $lead->forceDelete();
                return redirect()->back()->withErrors(['msg' => 'This Phone Number is Duplicated']);
            }

            $user = auth()->user();
            activity()
                ->performedOn($lead)
                ->causedBy($user)
                ->event('Create Lead')
                ->log('User(' . $user->name . ') created lead (' . $lead->name . ')');
            if ($lead->type == 'lead') {
                return redirect()->route('leads.index')->with('success', 'Lead Created Successfully');
            } else {
                return redirect()->route('CILs')->with('success', 'Cil Created Successfully');
            }
        } catch (Exception $e) {
            return redirect()->back()->withErrors(['msg' => $e->getMessage()]);
        }
    }

    public function edit($leadId)
    {
        $lead = Lead::findOrFail($leadId);
        // dd($lead);
        $phones = PhoneNumber::where('lead_id', $leadId)->get();
        $users = User::get();
        $countries = Country::all();
        $campaigns = Campaign::all();
        $brokers = Broker::all();

        // Log activity
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Edit')
            ->log(auth()->user()->name . ' viewed lead: ' . $lead->name . ' edit page');

        return view('leads.edit', compact('lead', 'users', 'phones', 'countries', 'campaigns', 'brokers'));
    }

    public function update($leadId, LeadRequest $request)
    {
        // Update Lead Data
        $lead = Lead::findOrFail($leadId);

        $lead->name = $request->name;
        $lead->email = $request->email;
        $lead->job_title = $request->job_title;
        $lead->contact_tool = $request->contact_tool ?? null;
        $lead->reply_option = $request->reply_option ?? null;
        $lead->purpose = $request->purpose ?? null;
        $lead->segment = $request->segment ?? null;
        $lead->tag = $request->tag ?? null;
        $lead->note = $request->note ?? null;

        if ($lead->type == 'lead') {
            $lead->campaign_id = $request->campaign_id;
            $lead->source_id = $request->source_id ?? $lead->source_id;
            $lead->came_from = $request->came_from;

            $campaign = Campaign::find($request->campaign_id);
            if (isset($campaign)) {
                $lead->project_id = $campaign->project?->id ?? NULL;
            }
        } else {
            $lead->civil_id = $request->civil_id;
            $lead->broker_id = $request->broker_id;
        }

        $lead->update();

        // Log Activity
        $model = new Lead();
        $user = auth()->user();

        activity()
            ->performedOn($model)
            ->causedBy($user)
            ->event('Update Lead')
            ->log(auth()->user()->name . ' updated lead: ' . $lead->name);

        return redirect()->route('leads.show', $lead->id)->with('success', 'Lead Updated Successfully');
    }

    public function show($leadId)
    {
        $statuses = Status::get();
        $lead = Lead::findOrFail($leadId);

        $communications = Communication::where('lead_id', $leadId)->get();
        $lastUser = LeadUser::where('lead_id', $leadId)->latest('id')->first();
        $codes = Country::get();
        $leadData = Lead::findOrFail($leadId);

        $cs_role = Role::where('name', 'Sales Admin')->first();
        // dd($cs_role->users);

        $current_date = Carbon::now();
        // dd($current_date);
        // echo $mytime->toDateTimeString();

        $users = User::get();
        $CurrentUser = auth()->user();

        if ($CurrentUser->type != 'admin' && $CurrentUser->role->name != 'Marketing Team Leader' && $CurrentUser->role->name != 'Sales Admin') {
            $UserChildren = array();
            foreach ($users as $user) {
                // dd($user);
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }
        } else {
            $UserChildren = $users;
        }
        // Logs history
        $logs = ActivityLog::where(['subject_id' => $lead->id, 'subject_type' => 'App\Models\Lead'])->get();

        // string to array

        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Profile')
            ->log(auth()->user()->name . ' viewed lead: ' . $lead->name . ' profile');

        return view('leads.show', compact('lead', 'leadData', 'current_date', 'UserChildren', 'users', 'cs_role', 'lastUser', 'communications', 'statuses', 'codes', 'logs'));
        //streams
    }

    public function destroy($leadId)
    {
        $lead = Lead::findOrFail($leadId);
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Leads - Archive')
            ->log(auth()->user()->name . ' Archived lead: ' . $lead->name);

        $phoneNUmbers = PhoneNumber::where('lead_id', $lead->id)->get();
        foreach ($phoneNUmbers as $phoneNumber) {
            $phoneNumber->delete();
        }

        $lead->delete();

        return redirect()->route('leads.index')->with('success', 'Lead Archived Successfully');
    }

    public function importView()
    {
        return view('leads.import');
    }

    public function import(UploadLeadRequest $request)
    {
        if ($request->file) {
            Excel::import(new LeadsImport, $request->file);

            $model = new Lead();
            $CurrentUser = auth()->user();
            activity()
                ->performedOn($model)
                ->causedBy($CurrentUser)
                ->event('Leads - Import')
                ->log(auth()->user()->name . ' Imprted Leads Sheet');

            return redirect()->route('leads.index')->with('success', 'Lead Uploaded Successfully');
        } else {
            return redirect()->back()->with('error', 'Choose File Please');
        }
    }

    public function archive()
    {
        $leads = Lead::onlyTrashed()->get();

        // Log activity
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads archive')
            ->log(auth()->user()->name . ' viewed leads archive');

        return view('leads.archive', compact('leads'));
    }

    public function restore($leadId)
    {
        $lead = Lead::withTrashed()->findOrFail($leadId);
        $lead->restore();

        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Leads - Restore')
            ->log(auth()->user()->name . ' restored lead: ' . $lead->name);

        return redirect()->back()->with('success', 'Lead Restored Successfully');
    }

    public function restoreAll()
    {
        Lead::where(['type' => 'cil'])->onlyTrashed()->restore();

        $leads = new Lead();
        $user = auth()->user();

        activity()
            ->performedOn($leads)
            ->causedBy($user)
            ->event('Restore All buildings')
            ->log('All building Restored by ' . $user->name);

        return redirect()->back();
    }

    public function deletePermanently($id)
    {
        $lead = Lead::withTrashed()->find($id);
        $lead->forceDelete();

        return redirect()->route('leads.index')->with('success', 'Lead Trashed successfully');
    }

    public function bulkAction(Request $request, $leads)
    {
        // try{
        $leads = Lead::whereIn('id', explode(',', $leads))->get();
        $model = new Lead();
        $CurrentUser = auth()->user();

        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Leads Bulk-Action')
            ->log(auth()->user()->name . ' made a change into group of leads ');
        foreach ($leads as $lead) {

            if ($request->user_id) {
                $user = User::find($request->user_id);
                $lead->user_id = $request->user_id;
                $lead->status_id = 1;
                $lead->is_delay = 0;
                $lead->save();

                // Calculate Deadline
                $ExtraTime = Setting::where('key', 'ExtraTime')->first();
                $now = Carbon::now();
                $deadline = Carbon::parse($now)->addhours($ExtraTime->value);

                // Store into lead_user table -> we removed the many to many relation
                // we using it to store and manage the deadline of the user and withdrow leads
                $LeadUser = new LeadUser();
                $LeadUser->user_id = $request->user_id;
                $LeadUser->lead_id = $lead->id;
                $LeadUser->status = 'bending';
                $LeadUser->extraTime = $ExtraTime->value;
                $LeadUser->deadline = $deadline;
                $LeadUser->save();

                // Log Activity
                activity()
                    ->performedOn($lead)
                    ->causedBy($CurrentUser)
                    ->event('Leads Assign-user')
                    ->withProperties([
                        'action_creator_name' => $CurrentUser->name,
                        'action_creator_id' => $CurrentUser->id,
                        'action_type' => 'Re-Assign User to Lead',
                        'lead_name' => $lead->name,
                        'lead_id' => $lead->id,
                        'lead_status' => $lead->LeadStatus?->name ?? '',
                        'sales_id' => $user->name,
                        'sales_name' => $user->id,
                    ])
                    ->log($CurrentUser->name . 'Assigned Lead (' . $lead->name . ') to user (' . $user->name . ')');
            }
        }
        return redirect()->route('leads.index')->with('success', 'Lead Updated Successfully');
    }

    public function reAssignUser(Request $request, $leadId)
    {
        // Assign User
        $user = User::find($request->user_id);
        $lead = Lead::find($leadId);
        $lead->user_id = $user->id;
        $lead->status_id = 1;
        $lead->is_delay = 0;
        $lead->save();

        // Calculate Deadline
        $now = Carbon::now();
        $ExtraTime = Setting::where('key', 'ExtraTime')->first();
        $deadline = Carbon::parse($now)->addhours($ExtraTime->value);

        // Store into lead_user table -> we removed the many to many relation
        // we using it to store and manage the deadline of the user and withdrow leads
        $LeadUser = new LeadUser();
        $LeadUser->user_id = $request->user_id;
        $LeadUser->lead_id = $lead->id;
        $LeadUser->status = 'bending';
        $LeadUser->extraTime = $ExtraTime->value;
        $LeadUser->deadline = $deadline;
        $LeadUser->save();

        // Log Activity
        // $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($lead)
            ->causedBy($CurrentUser)
            ->event('Leads Assign-user')
            ->withProperties([
                'action_creator_name' => $CurrentUser->name,
                'action_creator_id' => $CurrentUser->id,
                'action_type' => 'Re-Assign User to Lead',
                'lead_name' => $lead->name,
                'lead_id' => $lead->id,
                'lead_status' => $lead->LeadStatus->name,
                'sales_name' => $user->name,
                'sales_id' => $user->id,
            ])
            ->log($CurrentUser->name . 'Assigned Lead (' . $lead->name . ') to user (' . $user->name . ')');

        return redirect()->route('leads.show', $leadId)->with('success', 'Lead Updated Successfully');
    }

    public function delayLeads()
    {

        $leads = Lead::where('is_delay', 1)->get();
        $CurrentUser = auth()->user();
        $users = User::get();
        if ($CurrentUser->type != 'admin') {
            $UserChildren = array();
            foreach ($users as $user) {
                // dd($user);
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }
        } else {
            $UserChildren = $users;
        }

        // Log activity
        $model = new Lead();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Delay')
            ->log(auth()->user()->name . ' viewed leads Delay');

        return view('leads.delayLeads', compact('leads', 'UserChildren'));
    }

    public function PhoneCode(Request $request)
    {
        $country = Country::find($request->country_id);
        $code = $country->phonecode;
        return json_encode(['code' => $code]);
    }

    public function campaignSources(Request $request)
    {

        $campaign = Campaign::find($request->campaign_id);
        $sources = $campaign->sources;
        return json_encode(['sources' => $sources]);
    }

    public function CILs(Request $request)
    {
        // Edit Activity Log
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - CILs')
            ->log(auth()->user()->name . ' visited CILs ');
        // end Activity Log

        if (auth()->user()->type == 'admin') {
            $leads = Lead::query();
            $leads->where(['is_delay' => 0, 'type' => 'cil']);
        } else {
            $users = User::get();
            $CurrentUser = auth()->user();
            $UserChildren = array();

            foreach ($users as $userr) {
                // dd($user);
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }

            $LeadBulk = array();
            if (count($CurrentUser->leads) != 0) {
                foreach ($CurrentUser->leads as $lead) {
                    array_push($LeadBulk, $lead->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $sales = User::find($UserChildren[$i]['id']);
                if (count($sales->leads) == 0) {
                    continue;
                } else {
                    foreach ($sales->leads as $lead) {
                        array_push($LeadBulk, $lead->id);
                    }
                }
            }
            $LeadBulk = array_unique($LeadBulk);
            $leads = Lead::query();
            $leads->where('is_delay', 0);
            $leads->whereIn('id', $LeadBulk);
        }
        $statuses = Status::get();
        // Filters
        $statuses = Status::get();
        $statusTypes = DB::table('statuses')->pluck('type')->toArray();
        $statusTypes = array_unique($statusTypes);
        $users = $creators = User::where('id', '!=', 1)->get();
        $campaigns = Campaign::all();
        $sources = Source::all();
        $projects = Project::all();
        $events = Event::with('communications')->get();
        $countries = Country::get();


        if ($request->user_id) {
            $leads->whereIn('user_id', $request->user_id);
        }

        if ($request->creator_id) {
            $leads->whereIn('created_by', $request->creator_id);
        }

        if ($request->status_id) {
            $leads->whereIn('status_id', $request->status_id);
        }

        if ($request->contact_tool) {
            $leads->whereIn('contact_tool', $request->contact_tool);
        }

        if ($request->campaign_id) {
            $leads->whereIn('campaign_id', $request->campaign_id);
        }

        if ($request->reply_option) {
            $leads->whereIn('reply_option', $request->reply_option);
        }

        if ($request->purpose) {
            $leads->whereIn('purpose', $request->purpose);
        }

        if ($request->segment) {
            $leads->whereIn('segment', $request->segment);
        }

        if ($request->tag) {
            $leads->whereIn('tag', $request->tag);
        }

        if ($request->time1 && $request->time2) {
            $leads->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $leads->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        if ($request->start_date && $request->end_date) {
            $leads->whereDate('confirmation_date', '>=', $request->start_date)
                ->whereDate('confirmation_date', '<=', $request->end_date);
        }

        $leads = $leads->paginate(100)->withQueryString();

        $CurrentUser = auth()->user();
        if ($CurrentUser->type != 'admin') {
            $UserChildren = array();
            foreach ($users as $user) {
                // dd($user);
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }

            // dd($UserChildren);
        } else {
            $UserChildren = $users;
        }

        // Max Number of phones
        $leadNumbers = 0;
        foreach ($leads as $lead) {
            $maxCount = count($lead->phone_numbers);
            // dd($maxCount);
            if ($maxCount > $leadNumbers) {
                $leadNumbers = $maxCount;
            }
        }

        // Edit Activity Log
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Index')
            ->log(auth()->user()->name . ' viewed leads index ');
        // end Activity Log
        return view('leads.index', compact('leads', 'users', 'countries', 'campaigns', 'UserChildren', 'projects', 'sources', 'statuses', 'creators', 'leadNumbers', 'statusTypes', 'events'));
    }

    public function confirmCIL(Request $request, $id)
    {
        // Edit Activity Log
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Confirm - CILs')
            ->log(auth()->user()->name . ' Confirmed CILs ');
        // end Activity Log

        $cil = Lead::find($id);
        $cil->confirmation_date = date("Y-m-d H:i:s");
        $cil->is_confirmed = 1;
        $cil->save();
        return redirect()->route('CILs')->with('success', 'Cil confirmed Successfully');
    }

    public function checkMobileNumber(Request $request)
    {
        $countries = Country::where('phonecode', $request->country_code)->get();
        $countryIds = $countries->pluck('id');

        $isExists = PhoneNumber::where('phone', $request->mobile_number)
            ->whereIn('country_id', $countryIds)
            ->exists();

        return response()->json([
            'exists' => $isExists
        ]);
    }


    public function getLeads()
    {
        //
        try {
            $leads = Lead::with('user', 'phone_numbers', 'communications')->get();

            // Create and return Excel file
            return Excel::download(new AllLeadsExport($leads), 'leads.xlsx');
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }
}
