<?php

namespace App\Http\Controllers;

use App\Http\Requests\UploadUnitRequest;
use App\Imports\UnitsImport;
use App\Imports\UnitsUpdateImport;
use App\Models\ActivityLog;
use App\Models\Batch;
use App\Models\Building;
use App\Models\Client;
use App\Models\ClientContract;
use App\Models\Contract;
use App\Models\Facility;
use App\Models\FacilityUnit;
use App\Models\File;
use App\Models\Gallery;
use App\Models\Installment;
use App\Models\Payment;
use App\Models\Phase;
use App\Models\Project;
use App\Models\Unit;
use App\Models\UnitPrice;
use App\Models\UnitSetting;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class UnitController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:units-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:units-read',['only' => ['show']]);
    //     $this->middleware('permission:units-update',['only' => ['edit', 'update']]);
    //     $this->middleware('permission:units-delete',['only' => ['destroy']]);
    // }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' accessed units index');

        // Default
        $units = Unit::query();
        $units->where('is_active', 1);

        if (auth()->user()->type == 'admin') {
            $Allunits = $units;
        } else {
            $Allunits = $units->whereHas('building.phase', function ($query) {
                $query->whereHas('users', function ($q) {
                    $q->where('users.id', auth()->user()->id);
                });
            });
        }

        // Filter
        $users = User::all();
        $projects = Project::get();
        $types = UnitSetting::whereNotIn('type', ['view', 'finishing'])->get();
        $typesFinishing = UnitSetting::where('type', 'finishing')->get();
        $typesView = UnitSetting::where('type', 'view')->get();

        if ($user->type != 'admin') {
            $Allunits->where(function ($query) use ($user) {
                if ($user->role->hasPermission('units-read-free')) {
                    $query->orWhere('status', 'free');
                }

                if ($user->role->hasPermission('units-read-reserved')) {
                    $query->orWhere('status', 'reserved');
                }

                if ($user->role->hasPermission('units-read-hold')) {
                    $query->orWhere('status', 'hold');
                }

                if ($user->role->hasPermission('units-read-downpayment')) {
                    $query->orWhere('status', 'downpayment');
                }

                if ($user->role->hasPermission('units-read-contracted')) {
                    $query->orWhere('status', 'taken');
                }

                if ($user->role->hasPermission('units-read-delivered')) {
                    $query->orWhere('status', 'delivered');
                }
            });
        }

        if ($request->user_id) {
            $Allunits->whereIn('user_id', $request->user_id);
        }

        if ($request->unit_id) {
            $Allunits->whereIn('id', $request->unit_id);
        }

        if ($request->building_id) {
            $Allunits->whereIn('building_id', $request->building_id);
        }

        if ($request->project_id) {
            $Allunits->where('project_id', $request->project_id);
        }

        if ($request->project_ids) {
            $Allunits->whereIn('project_id', $request->project_ids);
        }

        if ($request->status) {
            $Allunits->whereIn('status', $request->status);
        }

        if ($request->type) {
            $Allunits->whereIn('type', $request->type);
        }

        if ($request->finishing) {
            $Allunits->whereIn('finishing', $request->finishing);
        }

        if ($request->view) {
            $Allunits->whereIn('view', $request->view);
        }

        if ($request->floor) {
            $Allunits->whereIn('floor', $request->floor);
        }

        if ($request->bedrooms) {
            $Allunits->whereIn('bedrooms', $request->bedrooms);
        }

        if ($request->bathrooms) {
            $Allunits->whereIn('bathrooms', $request->bathrooms);
        }

        if ($request->dressing) {
            $Allunits->whereIn('dressing', $request->dressing);
        }

        if ($request->time1 && $request->time2) {
            $Allunits->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $Allunits->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        if ($request->bookedDate1 && $request->bookedDate2) {
            $Allunits->whereDate('Booked_date', '>=', $request->bookedDate1)
                ->whereDate('Booked_date', '<=', $request->bookedDate2);
        }

        if ($request->price1 && $request->price2) {
            $Allunits->where('unit_total_price', '>=', $request->price1)
                ->where('unit_total_price', '<=', $request->price2);
        }

        if ($request->space1 && $request->space2) {
            $Allunits->where('unit_space', '>=', $request->space1)
                ->where('unit_space', '<=', $request->space2);
        }

        $Allunits = $Allunits->paginate(100)->withQueryString();
        // $Allunits = $Allunits->get();
        return view('units.index', compact('Allunits', 'typesView', 'typesFinishing', 'types', 'users', 'projects'));
    }

    public function create()
    {
        //
        $building = Building::find($_GET['id']);
        $BuildingType = $building->type;

        $projects = Project::get();

        if ($BuildingType == 'Residential') {
            $types = UnitSetting::where('type', 'residential')->get();
        } elseif ($BuildingType == 'Commercial') {
            $types = UnitSetting::where('type', 'commercial')->get();
        } elseif ($BuildingType == 'Administrative') {
            $types = UnitSetting::where('type', 'administrative')->get();
        } elseif ($BuildingType == 'Medical') {
            $types = UnitSetting::where('type', 'medical')->get();
        } else {
            $types = UnitSetting::where(function ($query) {
                $query->where('type', '=', 'residential')
                    ->orWhere('type', '=', 'commercial')
                    ->orWhere('type', '=', 'medical')
                    ->orWhere('type', '=', 'administrative');
            })->get();
        }

        $views = UnitSetting::where('type', 'view')->get();
        $finishings = UnitSetting::where('type', 'finishing')->get();

        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' units creation page ');

        return view('units.create', compact('building', 'types', 'views', 'finishings', 'projects'));
    }

    public function store(Request $request)
    {
        $unit = new Unit;

        $building = Building::find($request->building_id);
        $project_id = $building->phase->project->id;
        $projectExists = Unit::where(['name' => $request->name, 'project_id' => $project_id])->get();

        if (count($projectExists) != 0) {
            return redirect()->back()
                ->with('error', 'unit name is taken.');
        }

        $unit->name = $request->name;
        $unit->floor = $request->floor;
        $unit->is_active = $request->is_active ? 1 : 0;
        $unit->bedrooms = $request->bedrooms ?? 0;
        $unit->bathrooms = $request->bathrooms ?? 0;
        $unit->dressing = $request->dressing ?? 0;
        // $unit->basement_space   = $request->basement ?? 0 ;
        $unit->description = $request->description;
        $unit->building_id = $request->building_id;
        $unit->garden_percentage = $request->garden_percentage ?? '';
        $unit->garden_space = $request->garden_space ?? '';
        $unit->roof_space = $request->roof_space ?? '';
        $unit->unit_space = $request->unit_space;

        $unit->type = $request->type;
        $unit->finishing = $request->finishing;
        $unit->view = $request->view;
        $unit->project_id = $project_id ?? '';

        if ($request->unit_meter_price != 0) {
            // Unit Price
            $unit->unit_meter_price = $request->unit_meter_price;
            $unit_price = $request->unit_space * $request->unit_meter_price;
            // Roof price
            $building = $unit->building;

            if (isset($request->roof_space)) {
                $roof_total_price = $request->unit_meter_price * $request->roof_space;

                $unit->roof_total_price = $request->has_roof ? $roof_total_price : 0;
                $roof_price = $unit->roof_total_price;
            } else {
                $roof_price = 0;
            }

            // garden price
            if (isset($request->garden_space)) {
                $garden_total_price = $request->unit_meter_price * $request->garden_space;

                $unit->garden_total_price = $request->floor == 0 ? $garden_total_price : 0;
                $garden_price = $unit->garden_total_price;
            } else {
                $garden_price = 0;
            }

            // unit price
            $unit->unit_total_price = $unit_price + $garden_price + $roof_price;
            $unit_total_price = $unit->unit_total_price;
        } else {
            $unit->unit_total_price = $request->unit_total_price;
            $unit->garden_total_price = 0;
            $unit->roof_total_price = 0;
        }
        $unit->save();


        // Store Cash Details
        $unitPrice = new UnitPrice();
        $unitPrice->unit_id = $unit->id;
        $unitPrice->meter_price = $request->unit_meter_price;
        $unitPrice->price = $request->unit_meter_price * $unit->unit_space;
        $unitPrice->years = 'cash';

        if (isset($request->roof_space)) {
            $roof_meter_price = $request->unit_meter_price;
            $roof_total_price = $roof_meter_price * $unit->roof_space;
            $roof_price = $roof_total_price;
            $unitPrice->roof_price = $request->has_roof ? $roof_price : 0;
        }

        if (isset($request->garden_space)) {
            $garden_meter_price = $request->unit_meter_price;
            $garden_total_price = $garden_meter_price * $unit->garden_space;
            $garden_price = $garden_total_price;
            $unitPrice->garden_price = $request->floor == 0 ? $garden_price : 0;
        }

        $unitPrice->total_price = $unitPrice->price + $roof_price + $garden_price;
        $unitPrice->save();


        // Calculate Other Prices
        for ($i = 0; $i < count($request->meter_price); $i++) {

            $unitPrice = new UnitPrice();
            $unitPrice->unit_id = $unit->id;
            $unitPrice->years = $request->years[$i];
            $unitPrice->meter_price = $request->meter_price[$i];
            $unitPrice->price = $request->meter_price[$i] * $unit->unit_space;

            if (isset($request->roof_space)) {
                $roof_meter_price = $request->meter_price[$i];
                $roof_total_price = $roof_meter_price * $unit->roof_space;
                $roof_price = $roof_total_price;
                $unitPrice->roof_price = $request->has_roof ? $roof_price : 0;
            }

            if (isset($request->garden_space)) {
                $garden_meter_price = $request->meter_price[$i];
                $garden_total_price = $garden_meter_price * $unit->garden_space;
                $garden_price = $garden_total_price;
                $unitPrice->garden_price = $request->floor == 0 ? $garden_price : 0;
            }

            $unitPrice->total_price = $unitPrice->price + $roof_price + $garden_price;
            $unitPrice->save();
        }


        $payment = new Payment();
        $payment->unit_id = $unit->id;
        $payment->save();

        //Manage Batches Depended on project->payment plan
        $project = Project::find($unit->project_id);

        foreach ($project->PaymentPlans as $plan) { //Every project has 2 plans
            foreach ($plan->PaymentPlanDetails as $details) {

                $batches = (int) $details->batch;
                for ($i = 0; $i < $batches; $i++) {

                    $batch = new Batch();
                    $batch->unit_id = $unit->id;
                    $batch->type = $plan->type;
                    $batch->title = $details->PaymentType->name;
                    $batch->date = $unit->created_at;
                    $batch->amount = $details->amount;
                    $batch->save();
                }
            }
        }

        return redirect()->route('units.show', $unit->id)
            ->with('success', 'unit created successfully.');
    }

    public function show(Unit $unit)
    {
        //
        $building = $unit->building;

        $FacilityUnits = FacilityUnit::where('unit_id', $unit->id)->get();
        $facilities_price = FacilityUnit::where('unit_id', $unit->id)->sum('price');

        $facilities = array();
        if (count($FacilityUnits) != 0) {
            foreach ($FacilityUnits as $f_unit) {
                $facility = Facility::find($f_unit->facility_id);

                array_push($facilities, [
                    'name' => $facility->name,
                    'price' => $f_unit->price,
                ]);
            }
        }

        // Contracts
        $payment = Payment::where('unit_id', $unit->id)->first();
        $unitContracts = array();

        if (isset($unit->contracts)) {
            foreach ($unit->contracts as $unitContract) {
                if (isset($unitContract->contract_id)) {
                    $contract = Contract::find($unitContract->contract_id);
                    array_push($unitContracts, [
                        'id' => $unitContract->id,
                        'name' => $contract->name . ' ' . $contract->type,
                        'is_delivered' => $unitContract->is_delivered,
                        'contract_date' => $unitContract->contract_date,
                    ]);
                }
            }
        }
        $clients = DB::table('clients')->select('id', 'name')->get();
        // Get History

        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' accsessed ' . $unit->name . 'page');
        return view('units.show', compact('unit', 'building', 'facilities', 'facilities_price', 'payment', 'clients', 'unitContracts'));
    }

    public function edit(Unit $unit)
    {

        $building = $unit->building;

        $views = UnitSetting::where('type', 'view')->get();
        $finishings = UnitSetting::where('type', 'finishing')->get();

        if ($building->type == 'Residential') {
            $types = UnitSetting::where('type', 'Residential')->get();
        } elseif ($building->type == 'Commercial') {
            $types = UnitSetting::where('type', 'Commercial')->get();
        } elseif ($building->type == 'Administrative') {
            $types = UnitSetting::where('type', 'administrative')->get();
        } else {
            $types = UnitSetting::where(function ($query) {
                $query->where('type', '=', 'Residential')
                    ->orWhere('type', '=', 'Commercial')
                    ->orWhere('type', '=', 'administrative');
            })->get();
        }

        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' accessed unit: ' . $unit->name . 'Edit page');
        return view('units.edit', compact('unit', 'building', 'types', 'finishings', 'views'));
    }

    public function update(Request $request, Unit $unit)
    {
        // Update unit basic information
        $unit->name = $request->name;
        $unit->floor = $request->floor;
        $unit->is_active = $request->is_active ? 1 : 0;
        $unit->bedrooms = $request->bedrooms ?? 0;
        $unit->bathrooms = $request->bathrooms ?? 0;
        $unit->dressing = $request->dressing ?? 0;
        $unit->basement_space = $request->basement;
        $unit->description = $request->description;
        $unit->unit_space = $request->unit_space;
        $unit->garden_percentage = $request->garden_percentage ?? '';
        $unit->garden_space = $request->garden_space ?? '';
        $unit->type = $request->type;
        $unit->finishing = $request->finishing;
        $unit->view = $request->view;

        // Calculate Unit Total Price
        if ($request->unit_meter_price != 0) {
            $unit->unit_meter_price = $request->unit_meter_price;
            $unit_price = $request->unit_space * $request->unit_meter_price;

            // Calculate roof price if applicable
            $roof_price = 0;
            if ($request->has_roof && isset($request->roof_space)) {
                $roof_total_price = $request->unit_meter_price * $request->roof_space;
                $unit->roof_total_price = $roof_total_price;
                $roof_price = $unit->roof_total_price;
            } else {
                $unit->roof_total_price = 0;
            }

            // Calculate garden price if applicable
            $garden_price = 0;
            if ($request->floor == 0 && isset($request->garden_space)) {
                $garden_total_price = $request->unit_meter_price * $request->garden_space;
                $unit->garden_total_price = $garden_total_price;
                $garden_price = $unit->garden_total_price;
            } else {
                $unit->garden_total_price = 0;
            }

            // Total price calculation
            $unit->unit_total_price = $unit_price + $roof_price + $garden_price;
        } else {
            $unit->unit_total_price = $request->unit_total_price;
            $unit->garden_total_price = 0;
            $unit->roof_total_price = 0;
        }
        $unit->save();

        // Delete existing prices to replace them with updated values
        foreach ($unit->prices as $price) {
            $price->delete();
        }

        // Store Cash Details
        $unitPrice = new UnitPrice();
        $unitPrice->unit_id = $unit->id;
        $unitPrice->meter_price = $request->unit_meter_price;
        $unitPrice->price = $request->unit_meter_price * $unit->unit_space;
        $unitPrice->years = 'cash';

        if (isset($request->roof_space)) {
            $roof_meter_price = $request->unit_meter_price;
            $roof_total_price = $roof_meter_price * $unit->roof_space;
            $roof_price = $roof_total_price;
            $unitPrice->roof_price = $request->has_roof ? $roof_price : 0;
        }

        if (isset($request->garden_space)) {
            $garden_meter_price = $request->unit_meter_price;
            $garden_total_price = $garden_meter_price * $unit->garden_space;
            $garden_price = $garden_total_price;
            $unitPrice->garden_price = $request->floor == 0 ? $garden_price : 0;
        }

        $unitPrice->total_price = $unitPrice->price + $roof_price + $garden_price;
        $unitPrice->save();

        // Calculate Other Prices after editing
        for ($i = 0; $i < count($request->meter_price); $i++) {
            $unitPrice = new UnitPrice();
            $unitPrice->unit_id = $unit->id;
            $unitPrice->years = $request->years[$i];
            $unitPrice->meter_price = $request->meter_price[$i];
            $unitPrice->price = $request->meter_price[$i] * $unit->unit_space;

            // Conditional Roof Price Calculation for each year
            $roof_price = 0;
            if ($request->has_roof && isset($request->roof_space)) {
                $roof_total_price = $request->meter_price[$i] * $request->roof_space;
                $unitPrice->roof_price = $roof_total_price;
                $roof_price = $unitPrice->roof_price;
            } else {
                $unitPrice->roof_price = 0;
            }

            // Conditional Garden Price Calculation for each year
            $garden_price = 0;
            if ($request->floor == 0 && isset($request->garden_space)) {
                $garden_total_price = $request->meter_price[$i] * $request->garden_space;
                $unitPrice->garden_price = $garden_total_price;
                $garden_price = $unitPrice->garden_price;
            } else {
                $unitPrice->garden_price = 0;
            }

            // Calculate the total price for each term
            $unitPrice->total_price = $unitPrice->price + $roof_price + $garden_price;

            // Save the updated UnitPrice
            $unitPrice->save();
        }

        return redirect()->route('units.show', $unit->id)
            ->with('success', 'Unit updated successfully.');
    }


    public function destroy(Unit $unit)
    {
        // dd('unit.destroy');
        $unit->delete();

        return redirect()->back()
            ->with('success', 'Unit deleted successfully');
    }

    public function hideUnit($id)
    {
        //
        // dd($id);
        $user = auth()->user();

        $unit = Unit::find($id);

        if ($unit->is_active == 0) {
            $unit->is_active = 1;
            activity()
                ->event('hide')
                ->log($user->name . ' Activated : ' . $unit->name);
        } else {
            $unit->is_active = 0;
            activity()
                ->event('hide')
                ->log($user->name . 'de-Activated : ' . $unit->name);
        }
        $unit->save();

        return redirect()->back()->with('success', $unit->is_active == 0 ? 'Unit has been hidden successfully' : 'Unit Activated Successfully');
    }

    public function changeStatus(Request $request)
    {
        // dd('');
        $unit = Unit::find($request->unit_id);
        $client = Client::find($unit->client_id);
        if ($request->status == "revoke") {

            //  Activity Log
            $user = auth()->user();
            $Model = new Unit();
            activity()
                ->performedOn($Model)
                ->causedBy($user)
                ->event('Unit Actions')
                ->withProperties([
                    'unit_id' => $unit->id,
                    'client_id' => $client->id,
                ])
                ->log('The unit (' . $unit->name . ') has been (Discarded) From Client ' . $client->name);

            $Model = new Client();
            activity()
                ->performedOn($Model)
                ->causedBy($user)
                ->event('Client Actions')
                ->withProperties([
                    'unit_id' => $unit->id,
                    'client_id' => $client->id,
                ])
                ->log('The Client (' . $client->name . ') Revoked the Unit (' . $unit->name . ')');

            $unit->client_id = null;
            $unit->Booked_date = null;
            $unit->status = "free";

            // Asking
            $unitContract = ClientContract::where('unit_id', $request->unit_id)->first();
            if ($unitContract != null) {
                $unitContract->delete();
            }
            $unitPayment = Payment::where('unit_id', $request->unit_id)->first();
            if ($unitPayment != null) {
                $unitPayment->delete();
            }
            $unitFacility = FacilityUnit::where('unit_id', $request->unit_id)->get();
            if (count($unitFacility) != 0) {
                foreach ($unitFacility as $u_facility) {
                    $u_facility->delete();
                }
            }

            // Discard Batches

            foreach ($unit->batches as $batch) {
                $batch->delete();
            }

            $project = $unit->building->phase->project;

            foreach ($project->PaymentPlans as $plan) { //Every project has 2 plans
                foreach ($plan->PaymentPlanDetails as $details) {

                    $batches = (int) $details->batch;
                    for ($i = 0; $i < $batches; $i++) {

                        $batch = new Batch();
                        $batch->unit_id = $unit->id;
                        $batch->type = $plan->type;
                        $batch->title = $details->PaymentType->name;
                        $batch->date = $unit->created_at;
                        $batch->amount = $details->amount;
                        $batch->save();
                    }
                }
            }
        }
        $unit->save();
        $user = auth()->user();
        activity()
            ->event('unit.change status')
            ->log($user->name . 'changed unit: ' . $unit->name . 'to' . $unit->status);

        return redirect()->back()->with('success', "the unit's status changed Successfully");
    }

    public function transfareUnit(Request $request)
    {
        $unit = Unit::find($request->unit_id);
        $client = Client::find($request->client_id);

        $unit->client_id = $client->id;
        $unit->save();

        // $batches = $unit->batches->where('status', 'unpaid')->get();
        // foreach($batches as $batch){
        //     $batch->client_id = $client->id;
        //     $batch->save();
        // }

        //  Activity Log
        $user = auth()->user();
        $Model = new Unit();
        activity()
            ->performedOn($Model)
            ->causedBy($user)
            ->event('Unit Actions')
            ->withProperties([
                'unit_id' => $unit->id,
                'client_id' => $client->id,
            ])
            ->log('The unit (' . $unit->name . ') has been (Transfared) To Client ' . $client->name);

        $Model = new Client();
        activity()
            ->performedOn($Model)
            ->causedBy($user)
            ->event('Client Actions')
            ->withProperties([
                'unit_id' => $unit->id,
                'client_id' => $client->id,
            ])
            ->log('The Client (' . $client->name . ') Owned the Unit (' . $unit->name . ')');

        return redirect()->back()->with('success', "Unit Transfares Successfully");
    }

    public function import(UploadUnitRequest $request)
    {
        try {
            if ($request->file) {
                Excel::import(new UnitsImport, $request->file);

                return redirect()->back()->with('success', 'Unit Uploaded Successfully');
            } else {
                return redirect()->back()->with('error', 'Choose File Please');
            }
        } catch (Exception $ex) {
            Log::info(["Error Uploading " => $ex->getMessage()]);
            return redirect()->back()->with('error', $ex->getMessage());
        }
        $user = auth()->user();
        activity()
            ->event('imported')
            ->log($user->name . 'attached file to import units');
    }

    public function UpdateSheetModal(UploadUnitRequest $request)
    {
        // dd($request);
        try {
            if ($request->file) {
                Excel::import(new UnitsUpdateImport, $request->file);

                return redirect()->back()->with('success', 'Units updated Successfully');
            } else {
                return redirect()->back()->with('error', 'Choose File Please');
            }
        } catch (Exception $ex) {
            Log::info(["Error Uploading " => $ex->getMessage()]);
            return redirect()->back()->with('error', $ex->getMessage());
        }
        $user = auth()->user();
        activity()
            ->event('imported')
            ->log($user->name . 'attached file to import updates for units');
    }

    public function fileUpload(Request $request)
    {
        // dd($request);
        $unit = unit::find($request->unit_id);

        if (isset($request['files'])) {

            for ($i = 0; $i < count($request['files']); $i++) {

                $fileName = time() . $request['files'][$i]->getClientOriginalName();
                $extension = $request['files'][$i]->extension();

                if ($extension == 'csv' || $extension == 'txt' || $extension == 'xlx' || $extension == 'xls' || $extension == 'pdf' || $extension == 'docx') {
                    // dd($extension);
                    $request['files'][$i]->move(public_path('files'), $fileName);

                    /* Store $fileName name in DATABASE from HERE */
                    $file = new File();
                    $file->unit_id = $request->unit_id;
                    $file->file = $fileName;
                    $file->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed file');
                }
            }
            return redirect()->route('units.show', $unit->id)->with('success', 'Files uploaded successfully');
        }

        if (isset($request['images'])) {

            for ($i = 0; $i < count($request['images']); $i++) {

                $imageName = time() . $request['images'][$i]->getClientOriginalName();
                $extension = $request['images'][$i]->extension();
                // dd($extension);
                if ($extension == 'png' || $extension == 'jpg' || $extension == 'jpeg') {
                    $request['images'][$i]->move(public_path('images'), $imageName);

                    /* Store $imageName name in DATABASE from HERE */
                    $image = new Gallery();
                    $image->unit_id = $request->unit_id;
                    $image->name = $imageName;
                    $image->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed image');
                }
            }
            return redirect()->route('units.show', $unit->id)->with('success', 'Files uploaded successfully');
        }
    }

    public function deliveryDate(Request $request)
    {

        $payment = Payment::find($request->payment_id);
        $payment->delivery_date = $request->delivery_date;
        $payment->save();

        if ($request->is_delivered == 1) {
            $unit = Unit::find($request->unit_id);
            $unit->status = 'delivered';
            $unit->save();
        }

        // // Log activity
        // $model = new Unit();
        // $CurrentUser = auth()->user();
        // activity()
        //     ->performedOn($model)
        //     ->causedBy($CurrentUser)
        //     ->event('Visit - units archive')
        //     ->log(auth()->user()->name .' viewed units archive');

        return redirect()->back()->with('success', 'Delivery updated successfully');
    }

    public function archive()
    {
        $units = Unit::onlyTrashed()->get();
        // Log activity
        $model = new Unit();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - units archive')
            ->log(auth()->user()->name . ' viewed units archive');

        return view('units.archive', compact('units'));
    }

    public function restore($unitId)
    {
        Unit::withTrashed()->find($unitId)->restore();
        $unit = Unit::find($unitId);

        $user = auth()->user();

        activity()
            ->performedOn($unit)
            ->causedBy($user)
            ->event('Restore unit')
            ->log('unit (' . $unit->name . ') Restored by ' . $user->name);

        return redirect()->back();
    }

    public function restoreAll()
    {
        Unit::onlyTrashed()->restore();

        $unit = new Unit();
        $user = auth()->user();

        activity()
            ->performedOn($unit)
            ->causedBy($user)
            ->event('Restore All units')
            ->log('All units Restored by ' . $user->name);

        return redirect()->back();
    }

    public function history($unitId)
    {
        $unit = Unit::find($unitId);
        $activities = ActivityLog::where(['subject_id' => $unit->id, 'subject_type' => 'App\Models\Unit'])->orderBy('created_at', 'desc')->get();
        return view('units.history', compact('activities'));
    }

    public function projectBuilding(Request $request)
    {
        $project = Project::find($request->project_id);

        $buildings = Building::whereHas('phase.project', function ($query) use ($project) {
            $query->where('id', $project->id);
        })->where('is_active', 1)->get();

        return json_encode(['buildings' => $buildings]);
    }

    public function projectUnit(Request $request)
    {
        $projects = Unit::where('project_id', $request->project_id)->get();

        if (!empty($projects)) {
            $project = Project::find($request->project_id);

            $projects = Unit::whereHas('building.phase.project', function ($query) use ($project) {
                $query->where('id', $project->id);
            })->get();
        }

        return json_encode(['projects' => $projects]);
    }

    public function unitsBuilding(Request $request)
    {
        $buildingIds = $request->building_id;
        $units = Unit::whereIn('building_id', $buildingIds)->where('status', '!=', 'free')->get();

        $units->each(function ($unit) {
            $unit->finishing = $unit->finishingLevel->name;
        });

        $uniqueUnits = $units->unique();

        return json_encode(['units' => $uniqueUnits]);
    }

    public function deleteImage($id)
    {
        Gallery::find($id)->delete();
        return redirect()->back();
    }

    public function unitDeliveryStatus($id)
    {
        // dd('projects');
        try {
            $unit = Unit::find($id);
            $todayDate = Carbon::now()->format('Y-m-d');
            // dd($todayDate);

            $unit->status = 'delivered';
            $unit->save();
            $payment = Payment::where('unit_id', $unit->id)->first();
            $payment->delivery_date = $todayDate;
            $payment->save();

            return redirect()->back()->with('success', 'unit Delivered Successfully.');
        } catch (Exception $e) {
            return redirect()->back()->with('Error', 'Failed to change unit status to be delivered');
        }
    }

    public function hiddenUnits(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' accessed units index');

        // Default
        $units = Unit::query();
        $units->where('is_active', 0);

        if (auth()->user()->type == 'admin') {
            $Allunits = $units;
        } else {
            $Allunits = $units->whereHas('building.phase', function ($query) {
                $query->whereHas('users', function ($q) {
                    $q->where('users.id', auth()->user()->id);
                });
            });
        }

        // Filter
        $users = User::all();
        $projects = Project::get();
        $types = UnitSetting::whereNotIn('type', ['view', 'finishing'])->get();
        $typesFinishing = UnitSetting::where('type', 'finishing')->get();
        $typesView = UnitSetting::where('type', 'view')->get();

        if ($user->type != 'admin') {
            $Allunits->where(function ($query) use ($user) {
                if ($user->role->hasPermission('units-read-free')) {
                    $query->orWhere('status', 'free');
                }

                if ($user->role->hasPermission('units-read-reserved')) {
                    $query->orWhere('status', 'reserved');
                }

                if ($user->role->hasPermission('units-read-hold')) {
                    $query->orWhere('status', 'hold');
                }

                if ($user->role->hasPermission('units-read-downpayment')) {
                    $query->orWhere('status', 'downpayment');
                }

                if ($user->role->hasPermission('units-read-contracted')) {
                    $query->orWhere('status', 'taken');
                }

                if ($user->role->hasPermission('units-read-delivered')) {
                    $query->orWhere('status', 'delivered');
                }
            });
        }

        if ($request->user_id) {
            $Allunits->whereIn('user_id', $request->user_id);
        }

        if ($request->unit_id) {
            $Allunits->whereIn('id', $request->unit_id);
        }

        if ($request->building_id) {
            $Allunits->whereIn('building_id', $request->building_id);
        }

        if ($request->project_id) {
            $Allunits->where('project_id', $request->project_id);
        }

        if ($request->project_ids) {
            $Allunits->whereIn('project_id', $request->project_ids);
        }

        if ($request->status) {
            $Allunits->whereIn('status', $request->status);
        }

        if ($request->type) {
            $Allunits->whereIn('type', $request->type);
        }

        if ($request->finishing) {
            $Allunits->whereIn('finishing', $request->finishing);
        }

        if ($request->view) {
            $Allunits->whereIn('view', $request->view);
        }

        if ($request->floor) {
            $Allunits->whereIn('floor', $request->floor);
        }

        if ($request->bedrooms) {
            $Allunits->whereIn('bedrooms', $request->bedrooms);
        }

        if ($request->bathrooms) {
            $Allunits->whereIn('bathrooms', $request->bathrooms);
        }

        if ($request->dressing) {
            $Allunits->whereIn('dressing', $request->dressing);
        }

        if ($request->time1 && $request->time2) {
            $Allunits->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $Allunits->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        if ($request->bookedDate1 && $request->bookedDate2) {
            $Allunits->whereDate('Booked_date', '>=', $request->bookedDate1)
                ->whereDate('Booked_date', '<=', $request->bookedDate2);
        }

        if ($request->price1 && $request->price2) {
            $Allunits->whereDate('unit_total_price', '>=', $request->price1)
                ->whereDate('unit_total_price', '<=', $request->price2);
        }

        if ($request->space1 && $request->space2) {
            $Allunits = $Allunits->whereBetween('unit_space', [$request->space1, $request->space2]);
        }

        $Allunits = $Allunits->paginate(100)->withQueryString();
        // $Allunits = $Allunits->get();
        return view('units.index', compact('Allunits', 'typesView', 'typesFinishing', 'types', 'users', 'projects'));
    }


    public function reservationReport(Request $request)
    {
        try {
            $unit = Unit::find($request->id);
            $unit->status = 'taken';
            $unit->payment->down_payment = $request->downPayment;
            $unit->save();

            $paymentPlan = $request->paymentPlan;
            $contractDate = Carbon::parse($request->contractDate);
            $paymentDay = $request->paymentDay;

            $installmentsPerYear = 0;

            // Retrieve unit payment data
            $unitPrice = $unit->payment->unitPrice;
            $totalPrice = $unit->payment->remaining - $unit->payment->mentainance;

            // Get the number of years for payment
            $years = $unitPrice->years;

            // Determine the number of installments
            if ($paymentPlan == 'quarter') {
                $installmentsPerYear = 4;  // Every 3 months
            } elseif ($paymentPlan == 'double-quarter') {
                $installmentsPerYear = 2;  // Every 6 months
            } elseif ($paymentPlan == 'annually') {
                $installmentsPerYear = 1;  // Every 12 months
            }

            // Calculate the total number of installments
            $totalInstallments = $installmentsPerYear * $years;

            // Calculate the amount per installment
            $installmentAmount = $totalPrice / $totalInstallments;

            for ($i = 1; $i <= $totalInstallments; $i++) {
                // Calculate the due date for each installment
                if ($i === 1) {
                    $installmentDate = $contractDate->copy();  // Start from the exact contract date
                } else {
                    // For subsequent installments, add months based on the payment plan
                    $installmentDate = $contractDate->copy()->addMonths(($i - 1) * (12 / $installmentsPerYear));
                }

                // Create a new Installment record
                Installment::create([
                    'unit_id' => $unit->id,
                    'amount' => $installmentAmount,
                    'due_date' => $installmentDate,
                    'remaining_balance' => $totalPrice - ($installmentAmount * $i),
                    'contract_date' => $contractDate,
                    'contract_day' => $paymentDay,
                    'is_paid' => false,
                    'paid_date' => null
                ]);
            }

            return view('units.report', compact('unit'));
        } catch (Exception $e) {
            dd($e->getMessage());
            return response()->json(['error' => 'An error occurred while calculating the payment plan.'], 500);
        }
    }

    public function showReservationReport($id)
    {
        try {
            $unit = Unit::find($id);

            return view('units.report', compact('unit'));
        } catch (Exception $e) {
            return response()->json(['error' => 'An error occurred while calculating the payment plan.'], 500);
        }
    }

    public function cancelUnit(Request $request, $id)
    {
        try {
            $unit = Unit::findOrFail($id);
            Installment::where('unit_id', $unit->id)->delete();

            $unit->status = 'free';
            $unit->client_id = null;
            $unit->Booked_date = null;

            // Save the changes
            $unit->save();

            // Return a success response
            return response()->json(['success' => true, 'message' => 'Unit status updated to free']);
        } catch (\Exception $e) {
            // Return an error response in case of failure
            return response()->json(['success' => false, 'message' => 'Failed to update unit status'], 500);
        }
    }
}
