<?php

namespace App\Services;

use Illuminate\Http\Request;
use App\Models\Status;
use App\Models\PhoneNumber;
use App\Models\Communication;

class LeadFilterService
{
    public function applyFilters(Request $request, $query)
    {
        // Extract relevant filters from the request
        $filters = $request->only([
            'user_id',
            'creator_id',
            'status_id',
            'statusTypes',
            'campaign_id',
            'project_id',
            'source_id',
            'tag',
            'segment',
            'purpose',
            'contact_tool',
            'reply_option',
            'name',
            'phone',
            'country_id',
            'lastActio_start_date',
            'lastActio_end_date',
            'time1',
            'time2',
            'date1',
            'date2'
        ]);

        // Apply filters using `when()` for cleaner logic
        $query->when($filters['user_id'] ?? null, fn($q, $userId) => $q->whereIn('user_id', $userId))
            ->when($filters['creator_id'] ?? null, fn($q, $creatorId) => $q->whereIn('created_by', $creatorId))
            ->when($filters['status_id'] ?? null, fn($q, $statusId) => $q->whereIn('status_id', $statusId))
            ->when($filters['statusTypes'] ?? null, function ($q, $statusTypes) {
                $statusIds = Status::whereIn('type', $statusTypes)->pluck('id');
                $q->whereIn('status_id', $statusIds);
            })
            ->when($filters['campaign_id'] ?? null, fn($q, $campaignId) => $q->whereIn('campaign_id', $campaignId))
            ->when($filters['project_id'] ?? null, fn($q, $projectId) => $q->whereIn('project_id', $projectId))
            ->when($filters['source_id'] ?? null, fn($q, $sourceId) => $q->whereIn('source_id', $sourceId))
            ->when($filters['tag'] ?? null, fn($q, $tag) => $q->whereIn('tag', $tag))
            ->when($filters['name'] ?? null, fn($q, $name) => $q->where('name', 'LIKE', "%$name%"))
            ->when($filters['phone'] ?? null, function ($q, $phone) {
                $leadIds = PhoneNumber::where('phone', 'LIKE', "%$phone%")->whereNotNull('lead_id')->pluck('lead_id');
                $q->whereIn('id', $leadIds);
            });

        // Apply filters for country_id
        if (!empty($filters['country_id'])) {
            $leadIds = PhoneNumber::whereIn('country_id', $filters['country_id'])->whereNotNull('lead_id')->pluck('lead_id');
            $query->whereIn('id', $leadIds);
        }

        // Apply filters for last action date
        if (!empty($filters['lastActio_start_date']) && !empty($filters['lastActio_end_date'])) {
            $communicationIds = Communication::whereBetween('created_at', [$filters['lastActio_start_date'], $filters['lastActio_end_date']])
                ->pluck('lead_id');
            $query->whereIn('id', $communicationIds);
        }

        // Apply date range filter
        if (!empty($filters['date1']) && !empty($filters['date2'])) {
            $query->whereBetween('created_at', [$filters['date1'], $filters['date2']]);
        }

        // Apply time range filter
        if (!empty($filters['time1']) && !empty($filters['time2'])) {
            $query->whereTime('created_at', '>=', $filters['time1'])
                ->whereTime('created_at', '<=', $filters['time2']);
        }

        return $query;
    }
}
