<?php

namespace App\Exports;

use App\Models\Broker;
use App\Models\User;
use App\Models\Agent;
use App\Models\PhoneNumber;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class BrokersExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $brokers = Broker::query();

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->user_id && !empty($this->request->user_id)) {
            $brokers->whereIn('user_id', $this->request->user_id);
            $hasFilters = true;
        }

        if ($this->request->broker_id && !empty($this->request->broker_id)) {
            $brokers->whereIn('id', $this->request->broker_id);
            $hasFilters = true;
        }

        if ($this->request->status && !empty($this->request->status)) {
            $brokers->whereIn('status', $this->request->status);
            $hasFilters = true;
        }

        if ($this->request->created_at_from && $this->request->created_at_to && 
            trim($this->request->created_at_from) !== '' && trim($this->request->created_at_to) !== '') {
            $brokers->whereBetween('created_at', [$this->request->created_at_from, $this->request->created_at_to]);
            $hasFilters = true;
        }

        if ($this->request->updated_at_from && $this->request->updated_at_to && 
            trim($this->request->updated_at_from) !== '' && trim($this->request->updated_at_to) !== '') {
            $brokers->whereBetween('updated_at', [$this->request->updated_at_from, $this->request->updated_at_to]);
            $hasFilters = true;
        }

        if ($this->request->name && trim($this->request->name) !== '') {
            $brokers->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        if ($this->request->email && trim($this->request->email) !== '') {
            $brokers->where('email', 'LIKE', '%' . trim($this->request->email) . '%');
            $hasFilters = true;
        }

        if ($this->request->type && trim($this->request->type) !== '') {
            $brokers->where('type', 'LIKE', '%' . trim($this->request->type) . '%');
            $hasFilters = true;
        }

        if ($this->request->owner_name && trim($this->request->owner_name) !== '') {
            $brokers->where('owner_name', 'LIKE', '%' . trim($this->request->owner_name) . '%');
            $hasFilters = true;
        }

        if ($this->request->authorized_name && trim($this->request->authorized_name) !== '') {
            $brokers->where('authorized_name', 'LIKE', '%' . trim($this->request->authorized_name) . '%');
            $hasFilters = true;
        }

        if ($this->request->commercial_record && trim($this->request->commercial_record) !== '') {
            $brokers->where('commercial_record', 'LIKE', '%' . trim($this->request->commercial_record) . '%');
            $hasFilters = true;
        }

        return $brokers->with(['phone_numbers.country', 'salesMan'])->get();
    }

    public function headings(): array
    {
        return [
            'Broker ID',
            'Name',
            'Email',
            'Type',
            'Description',
            'Address',
            'Owner Name',
            'Authorized Name',
            'Commercial Record',
            'Sales Admin',
            'Sales Man',
            'Status',
            'Has Contract',
            'Contract Start Date',
            'Contract End Date',
            'Commission',
            'Phone Number 1',
            'Phone Number 2',
            'Country',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($broker): array
    {
        return [
            $broker->id,
            $broker->name,
            $broker->email,
            $broker->type,
            $broker->description,
            $broker->address,
            $broker->owner_name,
            $broker->authorized_name,
            $broker->commercial_record,
            $broker->sales_admin,
            $broker->salesMan ? $broker->salesMan->name : '--',
            $broker->status,
            $broker->has_contract ? 'Yes' : 'No',
            $broker->contract_startDate ? \Carbon\Carbon::parse($broker->contract_startDate)->format('Y-m-d') : '--',
            $broker->contract_endDate ? \Carbon\Carbon::parse($broker->contract_endDate)->format('Y-m-d') : '--',
            $broker->commission,
            $broker->phone_numbers->count() > 0 ? $broker->phone_numbers[0]->phone : '--',
            $broker->phone_numbers->count() > 1 ? $broker->phone_numbers[1]->phone : '--',
            $broker->phone_numbers->count() > 0 ? $broker->phone_numbers[0]->country->name : '--',
            $broker->created_at ? $broker->created_at->format('Y-m-d H:i:s') : '--',
            $broker->updated_at ? $broker->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 