<?php

namespace App\Exports;

use App\Models\Campaign;
use App\Models\Project;
use App\Models\Lead;
use App\Models\Inquiry;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class CampaignsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $campaigns = Campaign::query();

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->project_id && !empty($this->request->project_id)) {
            $campaigns->whereIn('project_id', $this->request->project_id);
            $hasFilters = true;
        }

        if ($this->request->campaign_id && !empty($this->request->campaign_id)) {
            $campaigns->whereIn('id', $this->request->campaign_id);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $campaigns->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $campaigns->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->budget1 && $this->request->budget2 && 
            trim($this->request->budget1) !== '' && trim($this->request->budget2) !== '') {
            $campaigns->where('budget', '>=', $this->request->budget1)
                ->where('budget', '<=', $this->request->budget2);
            $hasFilters = true;
        }

        if ($this->request->status && trim($this->request->status) !== '') {
            $campaigns->where('status', $this->request->status);
            $hasFilters = true;
        }

        if ($this->request->name && trim($this->request->name) !== '') {
            $campaigns->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        if ($this->request->purpose && trim($this->request->purpose) !== '') {
            $campaigns->where('purpose', 'LIKE', '%' . trim($this->request->purpose) . '%');
            $hasFilters = true;
        }

        return $campaigns->with(['project', 'leads', 'inquiries'])->get();
    }

    public function headings(): array
    {
        return [
            'Campaign ID',
            'Name',
            'Description',
            'Status',
            'Start Date',
            'End Date',
            'Budget',
            'Target',
            'Purpose',
            'Project Name',
            'Total Leads',
            'Total Inquiries',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($campaign): array
    {
        $totalLeads = $campaign->leads->count();
        $totalInquiries = $campaign->inquiries->count();

        return [
            $campaign->id,
            $campaign->name,
            $campaign->description ?? '--',
            $campaign->status ?? '--',
            $campaign->start_date ? $campaign->start_date : '--',
            $campaign->end_date ? $campaign->end_date : '--',
            $campaign->budget ? number_format($campaign->budget, 2) : '--',
            $campaign->target ?? '--',
            $campaign->purpose ?? '--',
            $campaign->project ? $campaign->project->name : '--',
            $totalLeads,
            $totalInquiries,
            $campaign->created_at ? $campaign->created_at->format('Y-m-d H:i:s') : '--',
            $campaign->updated_at ? $campaign->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 