<?php

namespace App\Exports;

use App\Models\Communication;
use App\Models\User;
use App\Models\Lead;
use App\Models\CallAction;
use App\Models\Event;
use App\Models\Broker;
use App\Services\CommunicationService;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class CommunicationsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;
    protected $communicationService;

    public function __construct(Request $request)
    {
        $this->request = $request;
        $this->communicationService = new CommunicationService();
    }

    public function collection()
    {
        $communications = Communication::query();
        $filterData = $this->communicationService->getFilterData();
        
        $this->communicationService->applyFilters($communications, $this->request, $filterData);
        $this->communicationService->applyUserAccessControl($communications, $filterData['creators']);
        
        // Apply broker filter to get only lead communications
        $communications->where(['broker_id' => null]);

        return $communications->with([
            'lead.phone_numbers.country', 'lead.LeadStatus', 'user', 'callAction'
        ])->get();
    }

    public function headings(): array
    {
        return [
            'ID',
            'Lead Name',
            'Phone Numbers',
            'Lead Status',
            'Description',
            'Created Date',
            'Created By',
            'Call Action',
            'State',
            'Type'
        ];
    }

    public function map($communication): array
    {
        $phoneNumbers = '';
        if ($communication->lead && $communication->lead->phone_numbers) {
            $phones = [];
            foreach ($communication->lead->phone_numbers as $phone) {
                $phones[] = $phone->country->phonecode . ' ' . $phone->phone;
            }
            $phoneNumbers = implode(', ', $phones);
        }

        return [
            $communication->id,
            $communication->lead->name ?? '',
            $phoneNumbers,
            $communication->lead->LeadStatus->name ?? '',
            $communication->description ?? '',
            $communication->created_at ? $communication->created_at->format('Y-m-d H:i:s') : '',
            $communication->user->name ?? '',
            $communication->callAction->name ?? '',
            $communication->state ?? '',
            $communication->type ?? ''
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E2EFDA']
                ]
            ]
        ];
    }
}
