<?php

namespace App\Exports;

use App\Models\Ticket;
use App\Models\User;
use App\Models\Client;
use App\Models\Unit;
use App\Models\Project;
use App\Models\TicketType;
use App\Models\Method;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class DelayTicketsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $tickets = $this->buildDelayTicketQuery();

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->ticket_status && !empty($this->request->ticket_status)) {
            $tickets->whereIn('status', $this->request->ticket_status);
            $hasFilters = true;
        }

        if ($this->request->user_id && !empty($this->request->user_id)) {
            $tickets->whereIn('user_id', $this->request->user_id);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $tickets->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->call_date1 && $this->request->call_date2 && 
            trim($this->request->call_date1) !== '' && trim($this->request->call_date2) !== '') {
            $tickets->whereDate('call_date', '>=', $this->request->call_date1)
                ->whereDate('call_date', '<=', $this->request->call_date2);
            $hasFilters = true;
        }

        if ($this->request->ticketType_id && !empty($this->request->ticketType_id)) {
            $tickets->whereIn('ticket_type_id', $this->request->ticketType_id);
            $hasFilters = true;
        }

        if ($this->request->unit_id && !empty($this->request->unit_id)) {
            $tickets->whereIn('unit_id', $this->request->unit_id);
            $hasFilters = true;
        }

        if ($this->request->client_id && !empty($this->request->client_id)) {
            $tickets->whereIn('client_id', $this->request->client_id);
            $hasFilters = true;
        }

        if ($this->request->phone_number && trim($this->request->phone_number) !== '') {
            $tickets->whereHas('unit', function ($query) {
                $query->whereHas('client', function ($query) {
                    $query->whereHas('phone_numbers', function ($query) {
                        $query->where('phone', 'like', '%' . $this->request->phone_number . '%');
                    });
                });
            });
            $hasFilters = true;
        }

        if ($this->request->title && trim($this->request->title) !== '') {
            $tickets->where('title', 'LIKE', '%' . trim($this->request->title) . '%');
            $hasFilters = true;
        }

        if ($this->request->description && trim($this->request->description) !== '') {
            $tickets->where('description', 'LIKE', '%' . trim($this->request->description) . '%');
            $hasFilters = true;
        }

        return $tickets->with([
            'client', 'unit', 'project', 'assigned_to', 'ticket_created_by', 
            'ticket_type', 'method', 'comments', 'actions', 'files'
        ])->get();
    }

    private function buildDelayTicketQuery()
    {
        $tickets = Ticket::query();
        
        if (auth()->user()->type == 'admin') {
            return $tickets->where('delay', 1);
        }
        
        $currentUser = auth()->user();
        $userChildrenIds = $this->getUserChildrenIds($currentUser);
        $ticketIds = $this->getTicketIdsForUsers($currentUser, $userChildrenIds);
        
        return $tickets->where('delay', 1)->whereIn('id', $ticketIds);
    }

    private function getUserChildrenIds($currentUser): array
    {
        return User::all()
            ->filter(function($user) use ($currentUser) {
                return in_array($currentUser->id, $user->getParents()->pluck('id')->toArray());
            })
            ->pluck('id')
            ->toArray();
    }

    private function getTicketIdsForUsers($currentUser, $userChildrenIds)
    {
        return collect([$currentUser->id])
            ->merge($userChildrenIds)
            ->map(function($userId) {
                return User::find($userId)->user_tickets()->pluck('id');
            })
            ->flatten()
            ->toArray();
    }

    public function headings(): array
    {
        return [
            'Delay Ticket ID',
            'Title',
            'Description',
            'Status',
            'Priority',
            'Client Name',
            'Unit Name',
            'Project Name',
            'Assigned To',
            'Created By',
            'Ticket Type',
            'Method',
            'Call Date',
            'Delay Reason',
            'Total Comments',
            'Total Actions',
            'Total Files',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($ticket): array
    {
        $totalComments = $ticket->comments->count();
        $totalActions = $ticket->actions->count();
        $totalFiles = $ticket->files->count();

        return [
            $ticket->id,
            $ticket->title ?? '--',
            $ticket->description ?? '--',
            $ticket->status ?? '--',
            $ticket->priority ?? '--',
            $ticket->client ? $ticket->client->name : '--',
            $ticket->unit ? $ticket->unit->name : '--',
            $ticket->project ? $ticket->project->name : '--',
            $ticket->assigned_to ? $ticket->assigned_to->name : '--',
            $ticket->ticket_created_by ? $ticket->ticket_created_by->name : '--',
            $ticket->ticket_type ? $ticket->ticket_type->name : '--',
            $ticket->method ? $ticket->method->name : '--',
            $ticket->call_date ? $ticket->call_date : '--',
            $ticket->delay_reason ?? '--',
            $totalComments,
            $totalActions,
            $totalFiles,
            $ticket->created_at ? $ticket->created_at->format('Y-m-d H:i:s') : '--',
            $ticket->updated_at ? $ticket->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 