<?php

namespace App\Exports;

use App\Models\Unit;
use App\Models\User;
use App\Models\Project;
use App\Models\UnitSetting;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class UnitsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $user = auth()->user();
        $units = Unit::query();
        $units->where('is_active', 1);

        // Apply user access restrictions
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Operation Specialist') {
            $Allunits = $units;
        } else {
            $Allunits = $units->whereHas('building.phase', function ($query) {
                $query->whereHas('users', function ($q) {
                    $q->where('users.id', auth()->user()->id);
                });
            });
        }

        // Apply user permission restrictions
        if ($user->type != 'admin') {
            $Allunits->where(function ($query) use ($user) {
                if ($user->role->hasPermission('units-read-free')) {
                    $query->orWhere('status', 'free');
                }
        
                if ($user->role->hasPermission('units-read-reserved')) {
                    $query->orWhere('status', 'reserved');
                }
        
                if ($user->role->hasPermission('units-read-hold')) {
                    $query->orWhere('status', 'hold');
                }
        
                if ($user->role->hasPermission('units-read-downpayment')) {
                    $query->orWhere('status', 'downpayment');
                }
        
                if ($user->role->hasPermission('units-read-contracted')) {
                    $query->orWhere('status', 'taken');
                }
        
                if ($user->role->hasPermission('units-read-delivered')) {
                    $query->orWhere('status', 'delivered');
                }
            });
        }

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->user_id && !empty($this->request->user_id)) {
            $Allunits->whereIn('user_id', $this->request->user_id);
            $hasFilters = true;
        }

        if ($this->request->unit_id && !empty($this->request->unit_id)) {
            $Allunits->whereIn('id', $this->request->unit_id);
            $hasFilters = true;
        }

        if ($this->request->building_id && !empty($this->request->building_id)) {
            $Allunits->whereIn('building_id', $this->request->building_id);
            $hasFilters = true;
        }

        if ($this->request->project_id && trim($this->request->project_id) !== '') {
            $Allunits->where('project_id', $this->request->project_id);
            $hasFilters = true;
        }
        
        if ($this->request->project_ids && !empty($this->request->project_ids)) {
            $Allunits->whereIn('project_id', $this->request->project_ids);
            $hasFilters = true;
        }

        if ($this->request->status && !empty($this->request->status)) {
            $Allunits->whereIn('status', $this->request->status);
            $hasFilters = true;
        }

        if ($this->request->type && !empty($this->request->type)) {
            $Allunits->whereIn('type', $this->request->type);
            $hasFilters = true;
        }

        if ($this->request->finishing && !empty($this->request->finishing)) {
            $Allunits->whereIn('finishing', $this->request->finishing);
            $hasFilters = true;
        }

        if ($this->request->view && !empty($this->request->view)) {
            $Allunits->whereIn('view', $this->request->view);
            $hasFilters = true;
        }

        if ($this->request->floor && !empty($this->request->floor)) {
            $Allunits->whereIn('floor', $this->request->floor);
            $hasFilters = true;
        }

        if ($this->request->bedrooms && !empty($this->request->bedrooms)) {
            $Allunits->whereIn('bedrooms', $this->request->bedrooms);
            $hasFilters = true;
        }

        if ($this->request->bathrooms && !empty($this->request->bathrooms)) {
            $Allunits->whereIn('bathrooms', $this->request->bathrooms);
            $hasFilters = true;
        }

        if ($this->request->dressing && !empty($this->request->dressing)) {
            $Allunits->whereIn('dressing', $this->request->dressing);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $Allunits->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $Allunits->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->bookedDate1 && $this->request->bookedDate2 && 
            trim($this->request->bookedDate1) !== '' && trim($this->request->bookedDate2) !== '') {
            $Allunits->whereDate('Booked_date', '>=', $this->request->bookedDate1)
                ->whereDate('Booked_date', '<=', $this->request->bookedDate2);
            $hasFilters = true;
        }

        if ($this->request->price1 && $this->request->price2 && 
            trim($this->request->price1) !== '' && trim($this->request->price2) !== '') {
            $Allunits->where('unit_total_price', '>=', $this->request->price1)
                ->where('unit_total_price', '<=', $this->request->price2);
            $hasFilters = true;
        }
        
        if ($this->request->space1 && $this->request->space2 && 
            trim($this->request->space1) !== '' && trim($this->request->space2) !== '') {
            $Allunits->whereBetween('unit_space', [$this->request->space1, $this->request->space2]);
            $hasFilters = true;
        }

        return $Allunits->with(['building.phase.project', 'user', 'client', 'typee', 'finishingLevel', 'vieww'])->get();
    }

    public function headings(): array
    {
        return [
            'Unit ID',
            'Unit Name',
            'Unit Number',
            'Status',
            'Type',
            'Finishing',
            'View',
            'Floor',
            'Bedrooms',
            'Bathrooms',
            'Dressing',
            'Unit Space',
            'Unit Total Price',
            'Building Name',
            'Phase Name',
            'Project Name',
            'Assigned User',
            'Client Name',
            'Booked Date',
            'Created Date'
        ];
    }

    public function map($unit): array
    {
        return [
            $unit->id,
            $unit->name,
            $unit->unit_number,
            $unit->status,
            $unit->typee ? $unit->typee->name : '--',
            $unit->finishingLevel ? $unit->finishingLevel->name : '--',
            $unit->vieww ? $unit->vieww->name : '--',
            $unit->floor,
            $unit->bedrooms,
            $unit->bathrooms,
            $unit->dressing,
            $unit->unit_space,
            $unit->unit_total_price,
            $unit->building ? $unit->building->name : '--',
            $unit->building && $unit->building->phase ? $unit->building->phase->name : '--',
            $unit->building && $unit->building->phase && $unit->building->phase->project ? $unit->building->phase->project->name : '--',
            $unit->user ? $unit->user->name : '--',
            $unit->client ? $unit->client->name : '--',
            $unit->Booked_date ? \Carbon\Carbon::parse($unit->Booked_date)->format('Y-m-d') : '--',
            $unit->created_at ? $unit->created_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 