<?php

namespace App\Http\Controllers\Administration;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\CommunicationRequest;
use App\Models\CallAction;
use App\Models\Communication;
use App\Models\User;
use App\Models\Lead;
use App\Models\Broker;
use App\Models\Agent;
use App\Models\Event;
use App\Models\Status;
use App\Models\CommunicationEvent;
use Illuminate\Support\Facades\DB;
use Exception;
use App\Notifications\AssignUser;
use App\Models\Notifications;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use App\Services\CommunicationService;
use App\Exports\CommunicationsExport;
use Maatwebsite\Excel\Facades\Excel;

class CommunicationController extends Controller
{
    public function __construct()
    {
        // $this->middleware('permission:communications-create',['only' => ['create', 'store']]);
        // $this->middleware('permission:communications-read',['only' => ['show']]);
    }

    public function index(Request $request, CommunicationService $communicationService)
    {
        $communicationService->logActivity('Events Index');
        
        $communications = Communication::query();
        $filterData = $communicationService->getFilterData();
        
        $communicationService->applyFilters($communications, $request, $filterData);
        $communicationService->applyUserAccessControl($communications, $filterData['creators']);
        
        // Clone the query for total count to avoid memory issues
        $totalActionsQuery = clone $communications;
        $totalActions = $totalActionsQuery->count();
        
        // Apply broker filter and paginate
        $communications->where(['broker_id' => null]);
        $communications = $communications->paginate(50)->withQueryString();

        $events = $filterData['events'];
        $creators = $filterData['creators'];
        $callActions = $filterData['callActions'];
        $leads = $filterData['leads'];

        return view('communications.index', compact('communications', 'events', 'creators', 'callActions', 'leads', 'totalActions'));
    }

    public function create($leadId = null)
    {

        activity()
            ->event('visit')
            ->log(auth()->user()->name . ' is creating call action');

        $callActions = CallAction::all();
        $statuses = Status::get();
        // dd($statuses);
        if (auth()->user()->type == 'admin') {
            $leads = Lead::get();
            $users = User::where('id', '!=', auth()->user()->id)->get();
        } else {
            $leads = Lead::where('user_id', auth()->user()->id)->get();
            $users = '';
        }

        if (isset($leadId)) {
            $leadData = Lead::findOrFail($leadId);
            // dd($leadData);
            return view('communications.create', compact('leads', 'leadData', 'callActions', 'users', 'statuses'));
        }


        return view('communications.create', compact('leads', 'callActions', 'users'));
    }

    public function store(Request $request, CommunicationService $communicationService)
    {
        try {
            DB::beginTransaction();
            
            if ($request->broker_id) {
                // Broker Actions
                $communication = $communicationService->createBrokerCommunication($request->all());
                $communicationService->createBrokerEvent($communication, $request->all());
                $communicationService->logBrokerActivity($communication, $communication->broker);
            } else {
                // Lead Actions
                $communication = $communicationService->createLeadCommunication($request->all());
                $communicationService->createLeadEvent($communication, $request->all());
                $communicationService->logLeadActivity($communication, $communication->lead);
            }
            
            DB::commit();
            
            return redirect()->back()->with('success', 'Activity Created Successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }


    public function show($communicationId, CommunicationService $communicationService)
    {
        $data = $communicationService->getCommunicationShowData((int) $communicationId);
        
        $communicationService->logActivity(' viewed event');

        return view('communications.show', $data);
    }

    public function changeStatus($communicationId, Request $request)
    {
        $communication = Communication::find((int) $communicationId);
        activity()
            ->event('event.change status')
            ->log(auth()->user()->name . ' changed event status to ' . $request->status);

        $communication->state = $request->state;
        $communication->save();

        return redirect()->back()->with('success', 'Status Changed Successfully');
    }


    public function actionTypes(Request $request)
    {
        $actions = CallAction::where('type', $request->action)->get();
        return json_encode(['actionTypes' => $actions]);
    }

    public function nextActions(CommunicationService $communicationService)
    {
        $children = $communicationService->getUserChildrenForNextActions();

        $communications = Communication::whereIn('user_id', $children);
        $communications->whereNotNull('lead_id');
        $communications = $communications->paginate(100)->withQueryString();

        return view('communications.nextActions', compact('communications'));
    }

    public function myActions()
    {
        // dd('myAcyions');
        $CurrentUser = auth()->user();
        $communications = Communication::query(); 
        $communications->where('user_id' ,$CurrentUser->id);
        $communications->whereNotNull('lead_id')->get();
        $communications = $communications->paginate(100)->withQueryString();

        return view('communications.myActions', compact('communications'));
    }

    public function checkIn(Request $request, CommunicationService $communicationService)
    {
        $communication = Communication::findOrFail((int) $request->communication_id);
        
        $communicationService->checkInCommunication($communication, $request->all());
        $communicationService->logCheckInActivity($communication);

        return redirect()->back()->with('success', 'Activity Checked in Successfully');
    }

    public function update(Request $request, Communication $communication, CommunicationService $communicationService)
    {
        $communicationService->updateCommunication($communication, $request->all());
        $communicationService->updateCommunicationEvent($communication, $request->all());
        $communicationService->logCommunicationUpdate($communication);
        
        return redirect()->back()->with('success', 'Activity updated in Successfully');
    }

    public function activityReAssign(Request $request, CommunicationService $communicationService)
    {
        $communication = Communication::findOrFail((int) $request->communication_id);
        
        $communicationService->reassignCommunication($communication, $request->user_id);
        $communicationService->logReassignActivity($communication);

        return redirect()->back()->with('success', 'user changed successfully');
    }

    /**
     * Export communications to Excel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported communications data');

        return Excel::download(new CommunicationsExport($request), 'communications_' . date('Y-m-d_H-i-s') . '.xlsx');
    }
}
