<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Tag;
use App\Models\File;
use App\Models\Unit;
use App\Models\User;
use App\Models\Alert;
use App\Models\Action;
use App\Models\Client;
use App\Models\Method;
use App\Models\Ticket;
use App\Models\Project;
use App\Models\TicketType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use App\Http\Requests\TicketRequest;
use App\Exports\TicketsExport;
use App\Exports\DelayTicketsExport;
use Spatie\Activitylog\Models\Activity;
use Maatwebsite\Excel\Facades\Excel;

class TicketController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:tickets-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:tickets-read',['only' => ['show']]);
    //     $this->middleware('permission:tickets-delete',['only' => ['destroy']]);
    // }

    public function index(Request $request): \Illuminate\Contracts\View\View
    {
        $tickets = $this->buildTicketQuery();
        $this->applyTicketFilters($tickets, $request);
        
        $tickets = $tickets->paginate(50)->withQueryString();
        $filterData = $this->getFilterData();

        return view('tickets.index', compact('tickets') + $filterData);
    }

    private function buildTicketQuery()
    {
        $tickets = Ticket::query();
        
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == "Customer Service Manager") {
            return $tickets->where('delay', 0);
        }
        
        $currentUser = auth()->user();
        $userChildrenIds = $this->getUserChildrenIds($currentUser);
        $ticketIds = $this->getTicketIdsForUsers($currentUser, $userChildrenIds);
        
        return $tickets->where('delay', 0)->whereIn('id', $ticketIds);
    }

    private function getUserChildrenIds($currentUser): array
    {
        return User::all()
            ->filter(function($user) use ($currentUser) {
                return in_array($currentUser->id, $user->getParents()->pluck('id')->toArray());
            })
            ->pluck('id')
            ->toArray();
    }

    private function getTicketIdsForUsers($currentUser, $userChildrenIds)
    {
        return collect([$currentUser->id])
            ->merge($userChildrenIds)
            ->map(function($userId) {
                return User::find($userId)->user_tickets()->pluck('id');
            })
            ->flatten()
            ->toArray();
    }

    private function getFilterData()
    {
        return [
            'ticketTypes' => DB::table('ticket_types')->select('id', 'name')->get(),
            'units' => DB::table('units')->select('id', 'name')->where('status', '!=', 'free')->get(),
            'methods' => DB::table('methods')->select('id', 'name')->get(),
            'users' => DB::table('users')->select('id', 'name')->get(),
            'clients' => DB::table('clients')->select('id', 'name')->get(),
        ];
    }

    private function applyTicketFilters($tickets, Request $request)
    {
        if ($request->ticket_status) {
            $tickets->whereIn('status', $request->ticket_status);
        }

        if ($request->user_id) {
            $tickets->whereIn('user_id', $request->user_id);
        }

        if ($request->date1 && $request->date2) {
            $tickets->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        if ($request->call_date1 && $request->call_date2) {
            $tickets->whereDate('call_date', '>=', $request->call_date1)
                ->whereDate('call_date', '<=', $request->call_date2);
        }

        if ($request->ticketType_id) {
            $tickets->whereIn('ticket_type_id', $request->ticketType_id);
        }

        if ($request->unit_id) {
            $tickets->whereIn('unit_id', $request->unit_id);
        }

        if ($request->client_id) {
            $tickets->whereIn('client_id', $request->client_id);
        }

    }

    private function storeTicketFiles($files, $ticket)
    {
        foreach ($files as $file) {
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('images'), $fileName);

            File::create([
                'ticket_id' => $ticket->id,
                'file' => $fileName,
            ]);
        }
    }

    private function createTicketAction($ticket, $user)
    {
        $actionData = [
            'ticket_id' => $ticket->id,
            'client_id' => $ticket->client_id,
            'unit_id' => $ticket->unit_id,
            'user_id' => $user->id,
            'action_type' => 'Create',
            'details' => $user->name . " Created Ticket " . $ticket->id . " and assigned it to user " . ($ticket->assigned_to?->name ?? 'Unknown'),
        ];

        if ($ticket->user_id) {
            $assignedUser = User::find($ticket->user_id);
            $actionData['assigned_user_id'] = $assignedUser->id;
            $actionData['assigned_user_name'] = $assignedUser->name;
        }

        Action::create($actionData);
    }

    private function createTicketAlert($ticket, $user)
    {
        if ($ticket->user_id != $user->id) {
            Alert::create([
                'user_id' => $ticket->user_id,
                'ticket_id' => $ticket->id,
                'title' => 'assigned ticket',
                'body' => 'you has been assigned to new ticket ' . $ticket->id,
                'link' => App::make('url')->to('/tickets/' . $ticket->id),
            ]);
        }
    }

    private function logTicketActivity($ticket, $user)
    {
        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event('Create')
            ->log($user->name . ' created Ticket (' . $ticket->id . ') and assigned it to user ' . ($ticket->assigned_to?->name ?? 'Unknown'));
    }

    public function create(): \Illuminate\Contracts\View\View
    {

        $users = User::get();
        $projects = Project::where('is_active', 1)->get();
        $methods = Method::get();
        $types = TicketType::get();
        $tags = Tag::get();
        return view('tickets.create', compact('users', 'projects', 'methods', 'types', 'tags'));
    }

    public function store(Request $request): \Illuminate\Http\RedirectResponse
    {
        $ticket = new Ticket();
        $ticket->description = $request->description;
        $ticket->user_id = $request->user_id; //assigned to
        $ticket->unit_id = $request->unit_id;
        $ticket->project_id = $request->project_id;
        $ticket->method_id = $request->method_id;
        $ticket->ticket_type_id = $request->ticket_type_id;
        $ticket->created_by = auth()->user()->id; // creator of ticket

        $unit = Unit::find($request->unit_id);
        $ticket->client_id = $unit->client->id;

        $ticket_type = TicketType::find($request->ticket_type_id);
        $callDate = Carbon::create($request->call_date);

        $deadline = Carbon::create($request->call_date);
        $deadline = $deadline->addDays($ticket_type->hours);

        $ticket->call_date = $callDate;
        $ticket->deadline = $deadline;

        $ticket->save();

        // Attach user_id with ticket
        $ticket->users()->attach($request->user_id);

        // store tag into clients
        $client = Client::find($ticket->client_id);
        $client->tag_id = $request->tag_id;
        $client->save();

        // Calculate deadline

        $ticket->deadline = $deadline;
        $ticket->save();

        // store files
        if ($request->hasFile('files')) {
            $this->storeTicketFiles($request->file('files'), $ticket);
        }
        $user = auth()->user();

        $this->createTicketAction($ticket, $user);
        $this->createTicketAlert($ticket, $user);
        $this->logTicketActivity($ticket, $user);

        return redirect()->route('tickets.index')
            ->with('success', 'Ticket created successfully.');
    }

    public function show(Ticket $ticket): \Illuminate\Contracts\View\View
    {
        $this->markAlertsAsRead($ticket);
        $this->logTicketVisit($ticket);

        $data = [
            'ticket' => $ticket,
            'comments' => $ticket->comments,
            'actions' => $ticket->actions,
            'users' => User::all(),
        ];

        return view('tickets.show', $data);
    }

    private function markAlertsAsRead($ticket)
    {
        $ticket->alerts()
            ->where('user_id', auth()->user()->id)
            ->update(['open' => 1]);
    }

    private function logTicketVisit($ticket)
    {
        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' visited Ticket ' . $ticket->id . ' page');
    }

    public function edit(Ticket $ticket): \Illuminate\Contracts\View\View
    {
        return view('tickets.edit', [
            'ticket' => $ticket,
            'users' => User::all(),
        ]);
    }

    public function update(TicketRequest $request, Ticket $ticket): \Illuminate\Http\RedirectResponse
    {
        $ticket->update($request->validated());

        return redirect()->route('tickets.index')
            ->with('success', 'Ticket updated successfully');
    }

    public function destroy(Ticket $ticket): \Illuminate\Http\RedirectResponse
    {
        $ticket->delete();
        
        return redirect()->route('tickets.index')
            ->with('success', 'Ticket deleted successfully');
    }

    public function TicketChangeStatus(Request $request, $id)
    {
        $ticket = Ticket::findOrFail($id);
        $ticket->update(['status' => $request->status]);

        $user = auth()->user();
        $this->createStatusChangeAction($ticket, $user);
        $this->logStatusChangeActivity($ticket, $user);

        return redirect()->back()->with('success', 'ticket Status Changed To ' . $request->status . ' successfully');
    }

    private function createStatusChangeAction($ticket, $user)
    {
        $actionData = [
            'ticket_id' => $ticket->id,
            'client_id' => $ticket->client_id,
            'unit_id' => $ticket->unit_id,
            'user_id' => $user->id,
            'action_type' => 'Change Status',
            'details' => 'Ticket Status Changed To (' . $ticket->status . ') by ' . $user->name,
        ];

        if ($ticket->user_id) {
            $assignedUser = User::find($ticket->user_id);
            $actionData['assigned_user_id'] = $assignedUser->id;
            $actionData['assigned_user_name'] = $assignedUser->name;
        }

        Action::create($actionData);
    }

    private function logStatusChangeActivity($ticket, $user)
    {
        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event($ticket->status)
            ->log('Ticket Status Changed To (' . $ticket->status . ') by ' . $user->name);
    }

    public function projectUnits(Request $request): \Illuminate\Http\JsonResponse
    {
        $project = Project::findOrFail($request->project_id);
        
        $units = Unit::whereHas('building.phase.project', function ($query) use ($project) {
            $query->where('id', $project->id);
        })
        ->where('is_active', 1)
        ->where('status', '!=', 'free')
        ->get();

        return response()->json(['units' => $units]);
    }

    public function delayTickets(Request $request): \Illuminate\Contracts\View\View
    {
        $tickets = $this->buildDelayTicketQuery();
        $this->applyTicketFilters($tickets, $request);
        
        $tickets = $tickets->paginate(50)->withQueryString();
        $filterData = $this->getFilterData();

        return view('tickets.index', compact('tickets') + $filterData);
    }

    private function buildDelayTicketQuery()
    {
        $tickets = Ticket::query();
        
        if (auth()->user()->type == 'admin') {
            return $tickets->where('delay', 1);
        }
        
        $currentUser = auth()->user();
        $userChildrenIds = $this->getUserChildrenIds($currentUser);
        $ticketIds = $this->getTicketIdsForUsers($currentUser, $userChildrenIds);
        
        return $tickets->where('delay', 1)->whereIn('id', $ticketIds);
    }

    public function reAssignUser(Request $request, $id): \Illuminate\Http\RedirectResponse
    {
        $ticket = Ticket::findOrFail($id);
        $ticket->update(['user_id' => $request->user_id]);
        $ticket->users()->attach($request->user_id);

        $user = auth()->user();
        $assignedUser = User::find($request->user_id);
        
        $this->createReassignAction($ticket, $user, $assignedUser);
        $this->createReassignAlert($ticket, $user);
        $this->logReassignActivity($ticket, $user, $assignedUser);

        return redirect()->back()
            ->with('success', 'ticket Assigned To ' . $assignedUser->name . ' successfully');
    }

    private function createReassignAction($ticket, $user, $assignedUser)
    {
        Action::create([
            'ticket_id' => $ticket->id,
            'client_id' => $ticket->client_id,
            'unit_id' => $ticket->unit_id,
            'user_id' => $user->id,
            'action_type' => 'Assign-User',
            'details' => $user->name . ' Assigned Ticket (' . $ticket->id . ') To user (' . $assignedUser->name . ')',
            'assigned_user_id' => $assignedUser->id,
            'assigned_user_name' => $assignedUser->name,
        ]);
    }

    private function createReassignAlert($ticket, $user)
    {
        if ($ticket->user_id != $user->id) {
            Alert::create([
                'user_id' => $ticket->user_id,
                'ticket_id' => $ticket->id,
                'title' => 'assigned ticket',
                'body' => 'you has been assigned to new ticket ' . $ticket->id,
                'link' => App::make('url')->to('/tickets/' . $ticket->id),
            ]);
        }
    }

    private function logReassignActivity($ticket, $user, $assignedUser)
    {
        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event("Assign-User")
            ->log($user->name . ' Assigned Ticket (' . $ticket->id . ') To user (' . $assignedUser->name . ')');
    }

    /**
     * Export tickets to Excel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported tickets data');

        return Excel::download(new TicketsExport($request), 'tickets_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export delay tickets to Excel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportDelayTickets(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported delay tickets data');

        return Excel::download(new DelayTicketsExport($request), 'delay_tickets_' . date('Y-m-d_H-i-s') . '.xlsx');
    }
}
