<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Permission;
use App\Models\Team;
use App\Models\Country;
use App\Models\PhoneNumber;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Password;
use App\Services\UserService;
use Maatwebsite\Excel\Facades\Excel;

use function PHPUnit\Framework\isEmpty;

class UserController extends Controller
{
    protected $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    // public function __construct()
    // {
    //     $this->middleware('permission:users-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:users-read',['only' => ['show']]);
    //     $this->middleware('permission:users-update',['only' => ['edit', 'update']]);
    //     $this->middleware('permission:users-delete',['only' => ['destroy']]);

    // }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function index()
    {
        $CurrentUser = auth()->user();
        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event('Visit Users Index')
            ->log($CurrentUser->name . ' viewed users index');

        $UserChildren = $this->userService->getUsersForCurrentUser($CurrentUser);

        return view('users.index', compact('UserChildren'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
        dd('something');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        dd($request);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        $userPermissions = $this->userService->getUserPermissions($user);

        $CurrentUser = auth()->user();
        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event("Visit User's Profile")
            ->log('User (' . $CurrentUser->name . ') Visited the profile of user (' . $user->name . ') ');

        return view('users.show', compact('user', 'userPermissions'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user)
    {
        $CurrentUser = auth()->user();
        $UserChildren = $this->userService->getUserChildrenForEdit($CurrentUser);

        $roles = Role::get();
        $countries = Country::all();

        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event("Visit User's Edit Page")
            ->log('User (' . $CurrentUser->name . ') Visited the Edit Page of user (' . $user->name . ') ');

        return view('users.edit', compact('user', 'countries', 'UserChildren', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user)
    {
        $this->userService->updateUserBasicInfo($user, $request);
        $this->userService->updateUserPhoneNumbers($user, $request);
        $this->userService->updateUserPermissions($user, $request);

        $userModel = new User();
        $CurrentUser = auth()->user();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event('Update User Profile')
            ->log('User (' . $CurrentUser->name . ') Updated Profile of user (' . $user->name . ') ');

        return redirect()->route('users.show', $user->id)
            ->with('success', 'user updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user, $id)
    {
        $user = User::find($id);
        activity()
            ->event('delete')
            ->log(auth()->user()->name . 'deleted user: ' . $user->name);

        $user->delete();
        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully');
    }

    public function profile()
    {
        $id = Auth::user()->id;
        $userType = $this->userService->getUserType($id);
        return view('auth.profile', compact('userType'));
    }

    public function editProfile(User $user)
    {
        return view('auth.editProfile', compact('user'));
    }

    public function updateProfile(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required',
            'password' => ['nullable', 'confirmed', 'min:6']
        ]);

        $this->userService->updateUserProfile($user, $request);

        return redirect()->route('profile')
            ->with('success', 'Profile updated successfully');
    }


    public function block($id)
    {
        $user = $this->userService->blockUser($id);

        $CurrentUser = auth()->user();
        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event("Blocking User")
            ->log('User (' . $CurrentUser->name . ') Blocked user (' . $user->name . ') ');

        return redirect()->route('users.index')
            ->with('success', 'user blocked successfully');
    }

    public function blocklist()
    {
        $users = $this->userService->getBlockedUsers();

        $user = auth()->user();
        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($user)
            ->event('Visit Users BlockList')
            ->log('User (' . $user->name . ') Visited Users  BlockList Page ');

        return view('users.blocklist', compact('users'));
    }

    public function unblock($id)
    {
        $user = $this->userService->unblockUser($id);

        $CurrentUser = auth()->user();
        $userModel = new User();

        activity()
            ->performedOn($userModel)
            ->causedBy($CurrentUser)
            ->event("unBlock User")
            ->log('User (' . $CurrentUser->name . ') un-Blocked user (' . $user->name . ') ');

        return redirect()->route('users.index')
            ->with('success', 'user blocked successfully');
    }

    public function usersTree()
    {
        $data = $this->userService->getUsersTreeData();
        $userModel = new User();
        $user = auth()->user();

        activity()
            ->performedOn($userModel)
            ->causedBy($user)
            ->event('Visit Users Tree')
            ->log('User (' . $user->name . ') Visited Users Tree Page ');

        return view('users.tree', $data);
    }

    public function export(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported users data');

        return Excel::download(new \App\Exports\UsersExport($request), 'users_' . date('Y-m-d_H-i-s') . '.xlsx');
    }
}
