<?php

namespace App\Imports;

use App\Models\Lead;
use App\Models\User;
use App\Models\Country;
use App\Models\Setting;
use App\Models\LeadUser;
use App\Models\PhoneNumber;
use Illuminate\Support\Facades\Session;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;

class LeadsImport implements ToModel, WithHeadingRow, WithValidation, SkipsEmptyRows, SkipsOnFailure
{
    use Importable, SkipsFailures;

    protected $invalidCountryCodes = []; // To store non-existent country codes

    public function model(array $row)
    {
        // Validate primary country code
        $country = Country::where('phonecode', $row['country_code'])->first();
        if (!$country) {
            $this->invalidCountryCodes[] = $row['country_code'];
            return null; // Skip this row
        }

        // Create the lead
        $lead = new Lead([
            'name' => $row['name'],
            'email' => $row['email'],
            'job_title' => $row['job_title'],
            'came_from' => $row['came_from'],
            'status_id' => $row['status_id'] ?? 1,
            'campaign_id' => $row['campaign_id'],
            'source_id' => $row['source_id'],
            'type' => 'lead',
            'project_id' => $row['project_id'],
            'purpose' => $row['purpose'],
            'segment' => $row['segment'],
            'tag' => $row['tag'],
            'contact_tool' => $row['contact_tool'],
            'reply_option' => $row['reply_option'],
            'note' => $row['note'],
            'user_id' => $row['user_id'] ?? null,
            'created_by' => auth()->user()->id,
        ]);
        $lead->save();

        // Create the primary phone number
        PhoneNumber::create([
            'phone' => $row['mobile_number'],
            'lead_id' => $lead->id,
            'country_id' => $country->id,
        ]);

        // Validate secondary country code if it exists
        if (!empty($row['mobile_number2'])) {
            $country2 = Country::where('phonecode', $row['country_code2'])->first();
            if (!$country2) {
                $this->invalidCountryCodes[] = $row['country_code2'];
            } else {
                PhoneNumber::create([
                    'phone' => $row['mobile_number2'],
                    'lead_id' => $lead->id,
                    'country_id' => $country2->id,
                ]);
            }
        }

        return $lead;
    }

    public function rules(): array
    {
        return [
            'name' => ['required', 'string'],
            'email' => ['nullable', 'unique:leads,email', 'email'],
            'country_code' => ['required'],
            'mobile_number' => ['required', 'unique:phone_numbers,phone'],
            'job_title' => ['nullable'],
            'came_from' => ['nullable'],
            'description' => ['nullable'],
            'campaign_id' => ['required'],
            'source_id' => ['required'],
            'project_id' => ['nullable'],
            'purpose' => ['nullable'],
            'segment' => ['nullable'],
            'note' => ['nullable'],
            'tag' => ['nullable'],
            'contact_tool' => ['nullable'],
            'reply_option' => ['nullable'],
            'user_id' => ['nullable'],
        ];
    }

    public function onFailure(\Maatwebsite\Excel\Validators\Failure ...$failures)
    {
        $data = [];
        foreach ($failures as $failure) {
            array_push($data, $failure->errors());
        }

        Session::push('leadsErrors', collect($data));
    }

    public function customValidationMessages()
    {
        return [
            'mobile_number.unique' => 'This mobile number :input is duplicated.',
            'mobile_number2.unique' => 'This mobile number :input is duplicated.',
        ];
    }

    public function getInvalidCountryCodes()
    {
        return array_unique($this->invalidCountryCodes); // Return unique invalid codes
    }
}
