<?php

namespace App\Services;

use App\Models\Lead;
use App\Models\User;
use App\Models\Status;
use App\Models\Campaign;
use App\Models\Country;
use App\Models\PhoneNumber;
use App\Models\LeadUser;
use App\Models\Setting;
use App\Models\ActivityLog;
use App\Models\Communication;
use App\Models\Event;
use App\Models\Source;
use App\Models\Project;
use App\Models\Broker;
use App\Models\Inquiry;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;
use Exception;

class LeadService
{
    public function logActivity(string $event, string $message, $subject = null): void
    {
        $model = $subject ?? new Lead();
        $currentUser = auth()->user();
        activity()->performedOn($model)->causedBy($currentUser)->event($event)->log($message);
    }

    public function buildLeadsQuery(): Builder
    {
        $leads = Lead::query();

        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Marketing Team Leader' || auth()->user()->role->name == 'Marketing Manager' || auth()->user()->role->name == 'Sales Admin' || auth()->user()->role->name == 'Operation Specialist') {
            $leads->where(['is_delay' => 0, 'type' => 'lead']);
        } elseif (auth()->user()->assigned_to == null) {
            $childUsers = auth()->user()->allChildren()->pluck('id')->toArray();
            if (count($childUsers) > 0) {
                $leads = Lead::where(function ($query) {
                    $query->where('created_by', auth()->user()->id)->orWhereIn('user_id', auth()->user()->allChildren()->pluck('id')->toArray());
                });
            } else {
                $leads = Lead::where('created_by', auth()->user()->id);
            }
        } else {
            $leads = $this->buildNonAdminLeadsQuery();
        }

        return $leads;
    }

    private function buildNonAdminLeadsQuery(): Builder
    {
        $users = User::all();
        $currentUser = auth()->user();
        $userChildren = $this->getUserChildrenArray($currentUser, $users);
        $leadBulk = $this->buildLeadBulk($currentUser, $userChildren);

        return Lead::query()->where('is_delay', 0)->whereIn('id', $leadBulk);
    }

    private function getUserChildrenArray($currentUser, $users): array
    {
        $userChildren = [];
        foreach ($users as $user) {
            $parents = $user->getParents()->pluck('id')->toArray();
            if (in_array($currentUser->id, $parents)) {
                $userChildren[] = [
                    'id' => $user->id,
                    'name' => $user->name,
                ];
            }
        }
        return $userChildren;
    }

    private function buildLeadBulk($currentUser, $userChildren): array
    {
        $leadBulk = [];

        // Add current user's leads
        if (count($currentUser->leads) != 0) {
            foreach ($currentUser->leads as $lead) {
                $leadBulk[] = $lead->id;
            }
        }

        // Add children's leads
        foreach ($userChildren as $child) {
            $sales = User::find($child['id']);
            if (count($sales->leads) != 0) {
                foreach ($sales->leads as $lead) {
                    $leadBulk[] = $lead->id;
                }
            }
        }

        return array_unique($leadBulk);
    }

    public function calculateMaxPhoneNumbers($leads): int
    {
        $leadNumbers = 0;
        foreach ($leads as $lead) {
            $maxCount = count($lead->phone_numbers);
            if ($maxCount > $leadNumbers) {
                $leadNumbers = $maxCount;
            }
        }
        return $leadNumbers;
    }

    public function getUserChildrenForLeads()
    {
        $currentUser = auth()->user();
        $users = User::all();

        if ($currentUser->type != 'admin' && $currentUser->role->name != 'Marketing Team Leader' && $currentUser->role->name != 'Sales Admin') {
            return $this->getUserChildrenArray($currentUser, $users);
        }

        return $users;
    }

    public function getFilterData(): array
    {
        return [
            'statuses' => Status::get(),
            'statusTypes' => \DB::table('statuses')->pluck('type')->unique()->toArray(),
            'users' => User::where('id', '!=', 1)->get(),
            'creators' => User::where('id', '!=', 1)->get(),
            'campaigns' => Campaign::all(),
            'sources' => Source::all(),
            'projects' => Project::all(),
            'countries' => Country::get(),
            'brokers' => Broker::get(),
        ];
    }

    public function createLead($data)
    {
        $lead = new Lead();
        $lead->name = $data['name'];
        $lead->email = $data['email'] ?? null;
        $lead->job_title = $data['job_title'] ?? null;
        $lead->contact_tool = $data['contact_tool'] ?? null;
        $lead->reply_option = $data['reply_option'] ?? null;
        $lead->purpose = $data['purpose'] ?? null;
        $lead->segment = $data['segment'] ?? null;
        $lead->tag = $data['tag'] ?? null;
        $lead->note = $data['note'];
        $lead->status_id = $data['status_id'];
        $lead->type = $data['type'];
        $lead->user_id = $data['user_id'] ?? null;
        $lead->created_by = auth()->user()->id;

        if ($data['type'] == 'lead') {
            $lead->campaign_id = $data['campaign_id'] ?? null;
            $lead->source_id = $data['source_id'] ?? null;
            $lead->came_from = $data['came_from'] ?? null;
            $campaign = Campaign::find($data['campaign_id']);
            if (isset($campaign)) {
                $lead->project_id = $campaign->project?->id ?? null;
            }
        } else {
            if (isset($data['confirmed']) && $data['confirmed'] == 'on') {
                $lead->confirmation_date = date('Y-m-d H:i:s');
                $lead->is_confirmed = 1;
            }
            $lead->civil_id = $data['civil_id'];
            $lead->broker_id = $data['broker_id'];
            $lead->agent_id = $data['agent_id'] ?? null;
        }

        if (isset($data['inquiry_id'])) {
            $lead->inquiry_id = $data['inquiry_id'];
            $inquiry = Inquiry::find($data['inquiry_id']);
            $inquiry->lead_id = $lead->id;
            $inquiry->save();
        }

        $lead->save();
        return $lead;
    }

    public function createPhoneNumbers(Lead $lead, array $mobileNumbers, array $countryIds): void
    {
        if (!empty($mobileNumbers)) {
            for ($i = 0; $i < count($mobileNumbers); $i++) {
                $phone = new PhoneNumber();
                $phone->lead_id = $lead->id;
                $phone->country_id = $countryIds[$i];
                $phone->phone = $mobileNumbers[$i];
                $phone->save();
            }
        }
    }

    public function assignUserToLead(Lead $lead, int $userId): void
    {
        $user = User::find($userId);
        $lead->user_id = $userId;
        $lead->status_id = 1;
        $lead->is_delay = 0;
        $lead->save();

        // Calculate Deadline
        $extraTime = Setting::where('key', 'ExtraTime')->first();
        $now = Carbon::now();
        $deadline = Carbon::parse($now)->addHours($extraTime->value);

        // Store into lead_user table
        $leadUser = new LeadUser();
        $leadUser->user_id = $userId;
        $leadUser->lead_id = $lead->id;
        $leadUser->status = 'bending';
        $leadUser->extraTime = $extraTime->value;
        $leadUser->deadline = $deadline;
        $leadUser->save();
    }

    public function deleteLeadPhoneNumbers(Lead $lead): void
    {
        $phoneNumbers = PhoneNumber::where('lead_id', $lead->id)->get();
        foreach ($phoneNumbers as $phoneNumber) {
            $phoneNumber->delete();
        }
    }

    public function getLeadShowData(int $leadId): array
    {
        $lead = Lead::findOrFail($leadId);
        $communications = Communication::where('lead_id', $leadId)->get();
        $lastUser = LeadUser::where('lead_id', $leadId)->latest('id')->first();
        $codes = Country::get();
        $leadData = Lead::findOrFail($leadId);
        $csRole = Role::where('name', 'Sales Admin')->first();
        $current_date = Carbon::now();
        $users = User::get();
        $UserChildren = $this->getUserChildrenForLeads();
        $logs = ActivityLog::where(['subject_id' => $lead->id, 'subject_type' => 'App\Models\Lead'])->get();

        return compact('lead', 'leadData', 'current_date', 'UserChildren', 'users', 'csRole', 'lastUser', 'communications', 'codes', 'logs');
    }

    public function getDelayLeadsData(Request $request = null): array
    {
        // get all leads where last communication is more than 24 hours
        $extraTime = Setting::where('key', 'ExtraTime')->first();
        $leads = Lead::whereHas('communications', function ($query) use ($extraTime) {
            $query->where('created_at', '<', Carbon::now()->subHours($extraTime->value));
        });

        // Apply user access restrictions
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Marketing Team Leader' || auth()->user()->role->name == 'Marketing Manager' || auth()->user()->role->name == 'Sales Admin' || auth()->user()->role->name == 'Operation Specialist') {
            $leads->where(['is_delay' => 0, 'type' => 'lead']);
        } elseif (auth()->user()->assigned_to == null) {
            $childUsers = auth()->user()->allChildren()->pluck('id')->toArray();
            if (count($childUsers) > 0) {
                $leads = Lead::where(function ($query) {
                    $query->where('created_by', auth()->user()->id)
                        ->orWhereIn('user_id', auth()->user()->allChildren()->pluck('id')->toArray());
                })->whereHas('communications', function ($query) use ($extraTime) {
                    $query->where('created_at', '<', Carbon::now()->subHours($extraTime->value));
                });
            } else {
                $leads = Lead::where('created_by', auth()->user()->id)->whereHas('communications', function ($query) use ($extraTime) {
                    $query->where('created_at', '<', Carbon::now()->subHours($extraTime->value));
                });
            }
        } else {
            $users = User::all();
            $CurrentUser = auth()->user();
            $UserChildren = array();

            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id');
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }

            $LeadBulk = array();
            if ($CurrentUser->leads()->count() != 0) {
                foreach ($CurrentUser->leads as $lead) {
                    array_push($LeadBulk, $lead->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $sales = User::find($UserChildren[$i]['id']);
                if (count($sales->leads) == 0) {
                    continue;
                } else {
                    foreach ($sales->leads as $lead) {
                        array_push($LeadBulk, $lead->id);
                    }
                }
            }
            $LeadBulk = array_unique($LeadBulk);
            $leads->where('is_delay', 0);
            $leads->whereIn('id', $LeadBulk);
        }

        // Paginate the results
        $perPage = $request ? $request->get('per_page', 50) : 50;
        $leads = $leads->with(['communications', 'phone_numbers.country', 'LeadStatus', 'source.sourceType', 'campaign', 'project', 'created_by_user'])->paginate($perPage);

        $currentUser = auth()->user();
        $users = User::get();
        $UserChildren = $this->getUserChildrenForLeads();

        return compact('leads', 'UserChildren');
    }

    public function confirmCIL(int $id): Lead
    {
        $cil = Lead::find($id);
        $cil->confirmation_date = date('Y-m-d H:i:s');
        $cil->is_confirmed = 1;
        $cil->save();
        return $cil;
    }

    public function checkMobileNumberExists(string $mobileNumber, string $countryCode): bool
    {
        $countries = Country::where('phonecode', $countryCode)->get();
        $countryIds = $countries->pluck('id');

        return PhoneNumber::where('phone', $mobileNumber)->whereIn('country_id', $countryIds)->exists();
    }

    public function getPhoneCode(int $countryId): string
    {
        $country = Country::find($countryId);
        return $country->phonecode;
    }

    public function getCampaignSources(int $campaignId): array
    {
        $campaign = Campaign::find($campaignId);
        return $campaign->sources->toArray();
    }
}
