<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Tag;
use App\Models\File;
use App\Models\Unit;
use App\Models\User;
use App\Models\Alert;
use App\Models\Action;
use App\Models\Client;
use App\Models\Method;
use App\Models\Ticket;
use App\Models\Project;
use App\Models\TicketType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use App\Http\Requests\TicketRequest;
use Spatie\Activitylog\Models\Activity;

class TicketController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:tickets-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:tickets-read',['only' => ['show']]);
    //     $this->middleware('permission:tickets-delete',['only' => ['destroy']]);
    // }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $tickets = Ticket::query();
        if (Auth()->user()->type == 'admin') {
            $tickets = Ticket::where('delay', 0);
            // dd($tickets->get());
        }
        else if(auth()->user()->role->name == "Customer Service Manager")
        {
            $tickets = Ticket::where('delay', 0);
        }
        else {
            $users = User::all();
            $CurrentUser = auth()->user();
            $UserChildren = array();
            // user children
            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }
            // tickets of user
            $TicketBulk = array();
            if (count($CurrentUser->user_tickets) != 0) {
                foreach ($CurrentUser->user_tickets as $ticket) {
                    array_push($TicketBulk, $ticket->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $user = User::find($UserChildren[$i]['id']);
                if (count($user->user_tickets) == 0) {
                    continue;
                } else {
                    foreach ($user->user_tickets as $ticket) {
                        array_push($TicketBulk, $ticket->id);
                    }
                }
            }

            $tickets->where('delay', 0);
            $tickets->whereIn('id', $TicketBulk);
        }

        // Filter Inputs
        $ticketTypes = DB::table('ticket_types')->select('id', 'name')->get();
        $units = DB::table('units')->select('id', 'name')->where('status', '!=', 'free')->get();
        $methods = DB::table('methods')->select('id', 'name')->get();
        $users = DB::table('users')->select('id', 'name')->get();
        $clients = DB::table('clients')->select('id', 'name')->get();

        // Start Filters

        // Direct Filter
        if ($request->ticket_status) {
            $tickets->whereIn('status', $request->ticket_status);
        }

        if ($request->user_id) {
            $tickets->whereIn('user_id', $request->user_id);
        }

        if ($request->date1 && $request->date2) {
            $tickets->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }
        if ($request->call_date1 && $request->call_date2) {
            $tickets->whereDate('call_date', '>=', $request->call_date1)
                ->whereDate('call_date', '<=', $request->call_date2);
        }

        if ($request->ticketType_id) {
            $tickets->whereIn('ticket_type_id', $request->ticketType_id);
        }

        if ($request->unit_id) {
            $tickets->whereIn('unit_id', $request->unit_id);
        }

        if ($request->client_id) {
            $tickets->whereIn('client_id', $request->client_id);
        }

        // End Filters
        $totalTickets = $tickets->get();
        $tickets = $tickets->paginate(50)->withQueryString();

        return view('tickets.index', compact('tickets', 'ticketTypes', 'units', 'methods', 'users', 'clients'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        $users = User::get();
        $projects = Project::where('is_active', 1)->get();
        $methods = Method::get();
        $types = TicketType::get();
        $tags = Tag::get();
        return view('tickets.create', compact('users', 'projects', 'methods', 'types', 'tags'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $ticket = new Ticket();
        $ticket->description = $request->description;
        $ticket->user_id = $request->user_id; //assigned to
        $ticket->unit_id = $request->unit_id;
        $ticket->project_id = $request->project_id;
        $ticket->method_id = $request->method_id;
        $ticket->ticket_type_id = $request->ticket_type_id;
        $ticket->created_by = auth()->user()->id; // creator of ticket

        $unit = Unit::find($request->unit_id);
        $ticket->client_id = $unit->client->id;

        $ticket_type = TicketType::find($request->ticket_type_id);
        $callDate = Carbon::create($request->call_date);

        $deadline = Carbon::create($request->call_date);
        $deadline = $deadline->addDays($ticket_type->hours);

        $ticket->call_date = $callDate;
        $ticket->deadline = $deadline;

        $ticket->save();

        // Attach user_id with ticket
        $ticket->users()->attach($request->user_id);

        // store tag into clients
        $client = Client::find($ticket->client_id);
        $client->tag_id = $request->tag_id;
        $client->save();

        // Calculate deadline

        $ticket->deadline = $deadline;
        $ticket->save();

        // store files
        if (isset($request['files'])) {
            for ($i = 0; $i < count($request['files']); $i++) {

                $imageName = time() . '.' . $request['files'][$i]->extension();
                //  dd($imageName);
                $request['files'][$i]->move(public_path('images'), $imageName);

                /* Store $imageName name in DATABASE from HERE */
                $file = new File();
                $file->ticket_id = $ticket->id;
                $file->file = $imageName;
                $file->save();
            }
        }
        $user = auth()->user();

        // Store Action
        $action = new Action();

        $action->ticket_id = $ticket->id;
        $action->client_id = $ticket->client_id;
        $action->unit_id = $ticket->unit_id;
        $action->user_id = $user->id;
        $action->action_type = 'Create';
        $action->details = $user->name . " Created Ticket " . ($ticket->id) . " and assigned it to user " . $ticket->assigned_to?->name;
        if (isset($ticket->user_id)) {
            $assigned_user = User::find($ticket->user_id);
            $action->assigned_user_id = $assigned_user->id;
            $action->assigned_user_name = $assigned_user->name;
        }

        $action->save();

        // Store Alert
        if ($ticket->user_id != auth()->user()->id) {
            $alert = new Alert();
            $alert->user_id = $ticket->user_id;
            $alert->ticket_id = $ticket->id;
            $alert->title = 'assigned ticket';
            $alert->body = 'you has been assigned to new ticket  ' . $ticket->id;
            $alert->link = App::make('url')->to('/tickets/' . $ticket->id);
            $alert->save();
        }

        // log activity
        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event('Create')
            ->log($user->name . ' created Ticket (' . $ticket->id . ') and assigned it to user ' . $ticket->name);

        return redirect()->route('tickets.index')
            ->with('success', 'Ticket created successfully.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function show(Ticket $ticket)
    {
        $alerts = $ticket->alerts()->where('user_id', auth()->user()->id)->get();

        foreach ($alerts as $alert) {
            $alert->open = 1;
            $alert->save();
        }

        $comments = $ticket->comments;
        $actions = $ticket->actions;
        $users = User::get();

        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' visited Ticket ' . $ticket->id . ' page ');
        return view('tickets.show', compact('ticket', 'comments', 'actions', 'users'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Ticket $ticket)
    {
        //
        $users = User::get();
        return view('tickets.edit', compact('ticket', 'users'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(TicketRequest $request, Ticket $ticket)
    {
        //
        $ticket->update($request->all());

        return redirect()->route('tickets.index')
            ->with('success', 'user Assigned successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Ticket $ticket)
    {
        //
        $ticket->delete();
        return redirect()->route('tickets.index')
            ->with('success', 'ticket Trashed successfully');
    }

    public function TicketChangeStatus(Request $request, $id)
    {
        //
        $ticket = Ticket::find($id);
        $ticket->status = $request->status;
        $ticket->save();

        $user = auth()->user();
        // Store Action
        $action = new Action();
        $action->ticket_id = $ticket->id;
        $action->client_id = $ticket->client_id;
        $action->unit_id = $ticket->unit_id;
        $action->user_id = $user->id; // action creator
        $action->action_type = 'Change Status';
        $action->details = 'Ticket Status Changed To (' . $ticket->status . ') by ' . $user->name;

        if (isset($ticket->user_id)) {
            $assigned_user = User::find($ticket->user_id);
            $action->assigned_user_id = $assigned_user->id;
            $action->assigned_user_name = $assigned_user->name;
        }
        $action->save();

        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event($ticket->status)
            ->log('Ticket Status Changed To (' . $ticket->status . ') by ' . $user->name);

        return redirect()->back()->with('success', 'ticket Status Changed To ' . $request->status . ' successfully');
    }

    public function projectUnits(Request $request)
    {
        $project = Project::find($request->project_id);
        $units = Unit::whereHas('building.phase.project', function ($query) use ($project) {
            $query->where('id', $project->id);
        })->where('is_active', 1)->where('status', '!=', 'free')->get();

        return json_encode(['units' => $units]);
    }

    public function delayTickets(Request $request)
    {
        //
        $tickets = Ticket::query();
        if (Auth()->user()->type == 'admin') {
            $tickets = Ticket::where('delay', 1);
        } else {
            $users = User::all();
            $CurrentUser = auth()->user();
            $UserChildren = array();
            // user children
            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }
            // tickets of user
            $TicketBulk = array();
            if (count($CurrentUser->user_tickets) != 0) {
                foreach ($CurrentUser->user_tickets as $ticket) {
                    array_push($TicketBulk, $ticket->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $user = User::find($UserChildren[$i]['id']);
                if (count($user->user_tickets) == 0) {
                    continue;
                } else {
                    foreach ($user->user_tickets as $ticket) {
                        array_push($TicketBulk, $ticket->id);
                    }
                }
            }

            $tickets->where('delay', 1);
            $tickets->whereIn('id', $TicketBulk);
        }

        // Filter Inputs
        $ticketTypes = DB::table('ticket_types')->select('id', 'name')->get();
        $units = DB::table('units')->select('id', 'name')->where('status', '!=', 'free')->get();
        $methods = DB::table('methods')->select('id', 'name')->get();
        $users = DB::table('users')->select('id', 'name')->get();
        $clients = DB::table('clients')->select('id', 'name')->get();

        // Start Filters

        if ($request->ticket_status) {
            $tickets->whereIn('status', $request->ticket_status);
        }

        if ($request->user_id) {
            $tickets->whereIn('user_id', $request->user_id);
        }

        if ($request->date1 && $request->date2) {
            $tickets->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }
        if ($request->call_date1 && $request->call_date2) {
            $tickets->whereDate('call_date', '>=', $request->call_date1)
                ->whereDate('call_date', '<=', $request->call_date2);
        }

        if ($request->ticketType_id) {
            $tickets->whereIn('ticket_type_id', $request->ticketType_id);
        }

        if ($request->unit_id) {
            $tickets->whereIn('unit_id', $request->unit_id);
        }

        if ($request->client_id) {
            $tickets->whereIn('client_id', $request->client_id);
        }

        // End Filters
        // $totalTickets = $tickets->get();
        $tickets = $tickets->paginate(50)->withQueryString();

        return view('tickets.index', compact('tickets', 'ticketTypes', 'units', 'methods', 'users', 'clients'));
    }

    public function reAssignUser(Request $request, $id)
    {
        //
        $ticket = Ticket::find($id);
        $ticket->user_id = $request->user_id;
        $ticket->save();

        $ticket->users()->attach($request->user_id);

        $user = auth()->user();
        // Store Action
        $action = new Action();
        $action->ticket_id = $ticket->id;
        $action->client_id = $ticket->client_id;
        $action->unit_id = $ticket->unit_id;
        $action->user_id = $user->id; // action creator
        $action->action_type = 'Assign-User';
        $action->details = $user->name . ' Assigned Ticket (' . $ticket->id . ') To user (' . $ticket->assigned_to->name . ')';

        if (isset($ticket->user_id)) {
            $assigned_user = User::find($ticket->user_id);
            $action->assigned_user_id = $assigned_user->id;
            $action->assigned_user_name = $assigned_user->name;
        }
        $action->save();
        // Store Alert
        if ($ticket->user_id != auth()->user()->id) {
            $alert = new Alert();
            $alert->user_id = $ticket->user_id;
            $alert->ticket_id = $ticket->id;
            $alert->title = 'assigned ticket';
            $alert->body = 'you has been assigned to new ticket ' . $ticket->id;
            $alert->link = App::make('url')->to('/tickets/' . $ticket->id);
            $alert->save();
        }

        activity()
            ->performedOn($ticket)
            ->causedBy($user)
            ->event("Assign-User")
            ->log($user->name . ' Assigned Ticket (' . $ticket->id . ') To user (' . $assigned_user->name . ')');

        return redirect()->back()
            ->with('success', 'ticket Assigned To ' . $assigned_user->name . ' successfully');
    }
}
